/**
 * WebView Bridge JavaScript 客户端
 * 提供与原生应用通信的接口
 */

class WebViewBridgeClient {
    constructor() {
        this.isConnected = false;
        this.messageId = 0;
        this.pendingCallbacks = new Map();
        
        // 自定义撤销栈系统
        this.undoStack = [];
        this.redoStack = [];
        this.maxUndoStackSize = 50; // 最大撤销步数
        this.isPerformingUndoRedo = false; // 防止在撤销/重做时记录状态
        
        this.init();
    }

    isEditableElement(element) {
        if (!element) return false;
        
        const tagName = element.tagName.toLowerCase();
        return (tagName === 'input' && element.type === 'text') || 
               tagName === 'textarea' || 
               element.contentEditable === 'true';
    }
    
    init() {
        // 检测平台并设置相应的通信方式
        this.detectPlatform();
        
        // 重置连接状态
        this.isConnected = false;
        
        // 设置消息接收处理器
        if (window.NativeBridge) {
            window.NativeBridge.onMessage(this.handleNativeMessage.bind(this));
            this.updateConnectionStatus(true);
        } else {
            // 等待NativeBridge准备就绪
            this.waitForBridge();
        }
        
        // 设置键盘事件监听器
        this.setupKeyboardEventListeners();
        
        // 页面加载完成后发送准备就绪消息
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => {
                this.sendMessage('pageReady', { url: window.location.href });
            });
        } else {
            this.sendMessage('pageReady', { url: window.location.href });
        }
    }
    
    detectPlatform() {
        const userAgent = navigator.userAgent;
        let platform = 'unknown';
        
        if (/iPhone|iPad|iPod/.test(userAgent)) {
            platform = 'iOS';
        } else if (/Macintosh/.test(userAgent)) {
            platform = 'macOS';
        } else if (/Android/.test(userAgent)) {
            platform = 'Android';
        } else if (/Windows/.test(userAgent)) {
            platform = 'Windows';
        }
        
        this.platform = platform;
        this.log(`检测到平台: ${platform}`);
    }
    
    waitForBridge() {
        let attempts = 0;
        const maxAttempts = 50; // 5秒超时
        
        const checkBridge = () => {
            if (window.NativeBridge) {
                window.NativeBridge.onMessage(this.handleNativeMessage.bind(this));
                this.updateConnectionStatus(true);
                this.log('桥接连接已建立');
            } else if (attempts < maxAttempts) {
                attempts++;
                setTimeout(checkBridge, 100);
            } else {
                this.log('桥接连接超时', 'error');
                this.updateConnectionStatus(false);
            }
        };
        
        checkBridge();
    }
    
    sendMessage(method, params = {}, callback = null) {
        if (!window.NativeBridge) {
            this.log(`发送消息失败: 桥接未准备就绪 - ${method}`, 'error');
            return;
        }
        
        const messageId = ++this.messageId;
        const message = {
            id: messageId,
            method: method,
            params: params,
            timestamp: Date.now()
        };
        
        if (callback) {
            this.pendingCallbacks.set(messageId, callback);
        }
        
        try {
            window.NativeBridge.sendMessage(method, params);
            this.log(`发送消息: ${method}`, 'sent', message);
        } catch (error) {
            this.log(`发送消息失败: ${error.message}`, 'error');
        }
    }
    
    handleNativeMessage(message) {
        try {
            const data = typeof message === 'string' ? JSON.parse(message) : message;
            this.log(`接收消息: ${data.method}`, 'received', data);
            
            // 处理响应消息
            if (data.method === 'response' && data.data && data.data.id) {
                const callback = this.pendingCallbacks.get(data.data.id);
                if (callback) {
                    callback(data.data);
                    this.pendingCallbacks.delete(data.data.id);
                }
                return;
            }
            
            // 处理特定消息类型
            switch (data.method) {
                case 'bridgeReady':
                    this.updateConnectionStatus(true);
                    break;
                case 'dataChanged':
                    this.handleDataChanged(data.params);
                    break;
                case 'statusChanged':
                    this.handleStatusChanged(data.params);
                    break;
                default:
                    this.handleCustomMessage(data);
            }
        } catch (error) {
            this.log(`处理消息失败: ${error.message}`, 'error');
        }
    }
    
    handleDataChanged(data) {
        this.log('数据已更改', 'info', data);
    }
    
    handleStatusChanged(data) {
        this.log(`状态已更改: ${data.status}`, 'info');
    }
    
    handleCustomMessage(data) {
        this.log(`自定义消息: ${data.method}`, 'info', data);
    }
    
    updateConnectionStatus(connected) {
        this.isConnected = connected;
        const statusElement = document.getElementById('status');
        if (statusElement) {
            statusElement.textContent = connected ? '已连接' : '未连接';
            statusElement.className = `status ${connected ? 'connected' : 'disconnected'}`;
        }
    }
    
    log(message, type = 'info', data = null) {
        const timestamp = new Date().toLocaleTimeString();
        const logElement = document.getElementById('messageLog');
        
        if (logElement) {
            const logEntry = `[${timestamp}] ${message}`;
            const fullEntry = data ? `${logEntry}\n${JSON.stringify(data, null, 2)}` : logEntry;
            
            logElement.textContent += fullEntry + '\n\n';
            logElement.scrollTop = logElement.scrollHeight;
        }
        
        // 同时输出到控制台
        console.log(`[WebViewBridge] ${message}`, data || '');
    }
    
    clearLog() {
        const logElement = document.getElementById('messageLog');
        if (logElement) {
            logElement.textContent = '';
        }
    }
    
    setupKeyboardEventListeners() {
        // 监听键盘事件
        document.addEventListener('keydown', this.handleKeyboardEvent.bind(this), true);
        document.addEventListener('keyup', this.handleKeyboardEvent.bind(this), true);
        
        // 确保页面可以接收键盘事件
        if (document.body) {
            document.body.setAttribute('tabindex', '0');
        }
        
        // 监听输入事件以自动保存撤销状态
        this.setupUndoRedoListeners();
        
        this.log('键盘事件监听器已设置', 'info');
    }

    setupUndoRedoListeners() {
        // 监听所有可能的编辑事件
        const events = ['input', 'paste', 'cut', 'keydown'];
        
        events.forEach(eventType => {
            document.addEventListener(eventType, (event) => {
                const element = event.target;
                
                // 只处理可编辑元素
                if (!this.isEditableElement(element)) {
                    return;
                }
                
                if (eventType === 'keydown') {
                    // 对于可能导致内容变化的按键，保存状态
                    if (this.isContentChangingKey(event)) {
                        this.saveStateBeforeEdit(element);
                    }
                } else if (eventType === 'input') {
                    // input 事件在内容已经改变后触发
                    this.saveStateAfterEdit(element);
                } else if (eventType === 'paste' || eventType === 'cut') {
                    // 粘贴和剪切操作
                    this.saveStateBeforeEdit(element);
                    setTimeout(() => this.saveStateAfterEdit(element), 10);
                }
            }, true); // 使用捕获阶段
        });
    }

    isContentChangingKey(event) {
        // 排除修饰键和导航键
        const key = event.key;
        const isModifierKey = event.ctrlKey || event.metaKey || event.altKey;
        
        // Cmd+Z 和 Cmd+Shift+Z 不应该触发状态保存
        if (isModifierKey && (key === 'z' || key === 'Z')) {
            return false;
        }
        
        // 其他修饰键组合（如 Cmd+C, Cmd+V 等）也不触发
        if (isModifierKey) {
            return false;
        }
        
        // 导航键不触发状态保存
        const navigationKeys = ['ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown', 'Home', 'End', 'PageUp', 'PageDown'];
        if (navigationKeys.includes(key)) {
            return false;
        }
        
        // 功能键不触发状态保存
        if (key.startsWith('F') && key.length > 1) {
            return false;
        }
        
        // 其他特殊键
        const specialKeys = ['Tab', 'Escape', 'CapsLock', 'Shift', 'Control', 'Alt', 'Meta'];
        if (specialKeys.includes(key)) {
            return false;
        }
        
        return true;
    }
    
    handleKeyboardEvent(event) {
        console.log('=== JavaScript handleKeyboardEvent 被调用 ===', {
            type: event.type,
            key: event.key,
            code: event.code,
            ctrlKey: event.ctrlKey,
            altKey: event.altKey,
            shiftKey: event.shiftKey,
            metaKey: event.metaKey,
            target: event.target ? event.target.tagName : 'no target',
            timestamp: new Date().toISOString()
        });
        
        // 检查是否是我们关心的快捷键组合
        const isCmdOrCtrl = event.metaKey || event.ctrlKey;
        
        if (!isCmdOrCtrl) {
            console.log('不是 Cmd/Ctrl 组合键，忽略');
            return; // 不是Cmd/Ctrl组合键，忽略
        }
        
        const key = event.key.toLowerCase();
        const supportedKeys = ['z', 'x', 'c', 'v', 'a'];
        
        if (!supportedKeys.includes(key)) {
            console.log(`不支持的键: ${key}`);
            return; // 不是我们支持的键，忽略
        }
        
        console.log(`处理快捷键: Cmd+${key.toUpperCase()}`);
        
        // 调用快捷键处理
        this.handleShortcut(event);
        
        // 记录键盘事件
        const eventInfo = {
            type: event.type,
            key: event.key,
            code: event.code,
            ctrlKey: event.ctrlKey,
            altKey: event.altKey,
            shiftKey: event.shiftKey,
            metaKey: event.metaKey,
            target: event.target ? event.target.tagName : 'unknown'
        };
        
        this.log(`键盘事件: ${event.type} - ${event.key}`, 'keyboard', eventInfo);
        
        // 处理特定的快捷键
        if (event.type === 'keydown') {
            this.handleShortcut(event);
        }
    }
    
    handleShortcut(event) {
        console.log('=== handleShortcut 被调用 ===');
        
        const key = event.key.toLowerCase();
        const isShift = event.shiftKey;
        const isMeta = event.metaKey;
        const isCtrl = event.ctrlKey;
        
        console.log(`快捷键详情: key=${key}, shift=${isShift}, meta=${isMeta}, ctrl=${isCtrl}, platform=${this.platform}`);
        
        // 在macOS上使用Cmd键，其他平台使用Ctrl键
        const isModifierPressed = (this.platform === 'macOS' && isMeta) || 
                                 (this.platform !== 'macOS' && isCtrl);
        
        if (!isModifierPressed) {
            console.log('修饰键检查失败');
            return;
        }
        
        let action = '';
        
        switch (key) {
            case 'z':
                action = isShift ? 'redo' : 'undo';
                break;
            case 'x':
                action = 'cut';
                break;
            case 'c':
                action = 'copy';
                break;
            case 'v':
                action = 'paste';
                break;
            case 'a':
                action = 'selectAll';
                break;
        }
        
        console.log(`确定的操作: ${action}`);
        
        if (action) {
            this.executeShortcutAction(action, event);
        } else {
            console.log('没有匹配的操作');
        }
    }
    
    executeShortcutAction(action, event) {
        console.log(`=== executeShortcutAction 被调用: ${action} ===`);
        
        // 阻止默认行为
        event.preventDefault();
        event.stopPropagation();
        
        // 获取当前焦点元素
        const activeElement = document.activeElement;
        console.log('当前焦点元素:', activeElement ? activeElement.tagName : 'none');
        
        // 检查是否在可编辑元素中
        if (this.isEditableElement(activeElement)) {
            console.log('在可编辑元素中执行操作');
            this.performEditAction(action, activeElement, event);
        } else {
            console.log('不在可编辑元素中，尝试查找可编辑元素');
            // 如果没有焦点元素，尝试查找页面中的可编辑元素
            const editableElements = document.querySelectorAll('input[type="text"], textarea, [contenteditable="true"]');
            if (editableElements.length > 0) {
                console.log(`找到 ${editableElements.length} 个可编辑元素，使用第一个`);
                this.performEditAction(action, editableElements[0], event);
            } else {
                console.log('页面中没有找到可编辑元素');
                this.log(`无法执行 ${action}：没有找到可编辑元素`, 'warning');
            }
        }
    }
    
    isEditableElement(element) {
        if (!element) return false;
        
        const tagName = element.tagName.toLowerCase();
        
        // 检查是否是输入元素
        if (tagName === 'input' || tagName === 'textarea') {
            return !element.disabled && !element.readOnly;
        }
        
        // 检查是否是可编辑的div
        if (element.contentEditable === 'true') {
            return true;
        }
        
        return false;
    }
    
    performEditAction(action, element, event) {
        console.log(`=== performEditAction 被调用: ${action} ===`);
        console.log('目标元素:', element ? element.tagName : 'none');
        
        if (!element) {
            console.log('没有目标元素，无法执行操作');
            return;
        }
        
        try {
            switch (action) {
                case 'undo':
                    console.log('执行撤销操作');
                    this.performUndo(element);
                    break;
                case 'redo':
                    console.log('执行重做操作');
                    this.performRedo(element);
                    break;
                case 'cut':
                    console.log('执行剪切操作');
                    this.saveStateBeforeEdit(element);
                    if (document.execCommand) {
                        document.execCommand('cut');
                    }
                    this.saveStateAfterEdit(element);
                    break;
                case 'copy':
                    console.log('执行复制操作');
                    if (document.execCommand) {
                        document.execCommand('copy');
                    }
                    break;
                case 'paste':
                    console.log('执行粘贴操作');
                    this.saveStateBeforeEdit(element);
                    // 粘贴操作需要特殊处理，因为现代浏览器限制了execCommand('paste')
                    this.handlePaste(element);
                    break;
                case 'selectAll':
                    console.log('执行全选操作');
                    if (element.select) {
                        element.select();
                    } else if (document.execCommand) {
                        document.execCommand('selectAll');
                    }
                    break;
                default:
                    console.log(`未知操作: ${action}`);
            }
            
            this.log(`成功执行操作: ${action}`, 'success');
            
            // 阻止默认行为，避免重复执行
            event.preventDefault();
            event.stopPropagation();
            
        } catch (error) {
            this.log(`执行操作失败: ${action} - ${error.message}`, 'error');
        }
    }
    
    async handlePaste(element) {
        try {
            // 检查是否支持Clipboard API
            if (navigator.clipboard && navigator.clipboard.readText) {
                // 添加权限检查
                const permission = await navigator.permissions.query({name: 'clipboard-read'});
                if (permission.state === 'granted' || permission.state === 'prompt') {
                    const text = await navigator.clipboard.readText();
                    this.insertTextAtCursor(element, text);
                    this.saveStateAfterEdit(element);
                    this.log('使用Clipboard API粘贴成功', 'success');
                    return;
                }
            }
            
            // 回退到execCommand
            if (document.execCommand) {
                document.execCommand('paste');
                this.saveStateAfterEdit(element);
                this.log('使用execCommand粘贴', 'info');
            } else {
                this.log('粘贴功能不可用', 'warning');
            }
        } catch (error) {
            this.log(`粘贴操作失败: ${error.message}`, 'error');
        }
    }
    
    insertTextAtCursor(element, text) {
        if (element.tagName.toLowerCase() === 'input' || element.tagName.toLowerCase() === 'textarea') {
            const start = element.selectionStart;
            const end = element.selectionEnd;
            const value = element.value;
            
            element.value = value.substring(0, start) + text + value.substring(end);
            element.selectionStart = element.selectionEnd = start + text.length;
        } else if (element.contentEditable === 'true') {
            const selection = window.getSelection();
            if (selection.rangeCount > 0) {
                const range = selection.getRangeAt(0);
                range.deleteContents();
                range.insertNode(document.createTextNode(text));
                range.collapse(false);
                selection.removeAllRanges();
                selection.addRange(range);
            }
        }
    }

    // 撤销/重做系统的核心方法
    saveStateBeforeEdit(element) {
        if (this.isPerformingUndoRedo || !element) {
            return;
        }

        const state = this.captureElementState(element);
        if (state) {
            // 清空重做栈，因为新的编辑操作会使重做无效
            this.redoStack = [];
            
            // 添加到撤销栈
            this.undoStack.push(state);
            
            // 限制撤销栈大小
            if (this.undoStack.length > this.maxUndoStackSize) {
                this.undoStack.shift();
            }
            
            this.log(`保存编辑前状态: ${element.tagName}`, 'debug');
        }
    }

    saveStateAfterEdit(element) {
        // 延迟保存，确保DOM已更新
        setTimeout(() => {
            if (this.isPerformingUndoRedo || !element) {
                return;
            }
            
            // 更新撤销栈中最后一个状态的"after"信息
            if (this.undoStack.length > 0) {
                const lastState = this.undoStack[this.undoStack.length - 1];
                const afterState = this.captureElementState(element);
                if (afterState) {
                    lastState.after = afterState.current;
                    this.log(`更新编辑后状态: ${element.tagName}`, 'debug');
                }
            }
        }, 10);
    }

    captureElementState(element) {
        if (!element) return null;

        const tagName = element.tagName.toLowerCase();
        
        if (tagName === 'input' || tagName === 'textarea') {
            return {
                element: element,
                tagName: tagName,
                current: {
                    value: element.value,
                    selectionStart: element.selectionStart,
                    selectionEnd: element.selectionEnd
                },
                after: null // 将在编辑后填充
            };
        } else if (element.contentEditable === 'true') {
            const selection = window.getSelection();
            let selectionInfo = null;
            
            if (selection.rangeCount > 0) {
                const range = selection.getRangeAt(0);
                selectionInfo = {
                    startOffset: range.startOffset,
                    endOffset: range.endOffset,
                    startContainer: range.startContainer,
                    endContainer: range.endContainer
                };
            }
            
            return {
                element: element,
                tagName: tagName,
                current: {
                    innerHTML: element.innerHTML,
                    selection: selectionInfo
                },
                after: null
            };
        }
        
        return null;
    }

    performUndo(element) {
        if (this.undoStack.length === 0) {
            this.log('没有可撤销的操作', 'info');
            return;
        }

        this.isPerformingUndoRedo = true;
        
        try {
            const state = this.undoStack.pop();
            
            // 保存当前状态到重做栈
            const currentState = this.captureElementState(state.element);
            if (currentState) {
                this.redoStack.push({
                    element: state.element,
                    tagName: state.tagName,
                    current: currentState.current,
                    after: state.current // 撤销后应该恢复到的状态
                });
            }
            
            // 恢复到之前的状态
            this.restoreElementState(state.element, state.current);
            
            this.log(`撤销操作成功: ${state.tagName}`, 'success');
            
        } catch (error) {
            this.log(`撤销操作失败: ${error.message}`, 'error');
        } finally {
            this.isPerformingUndoRedo = false;
        }
    }

    performRedo(element) {
        if (this.redoStack.length === 0) {
            this.log('没有可重做的操作', 'info');
            return;
        }

        this.isPerformingUndoRedo = true;
        
        try {
            const state = this.redoStack.pop();
            
            // 保存当前状态到撤销栈
            const currentState = this.captureElementState(state.element);
            if (currentState) {
                this.undoStack.push({
                    element: state.element,
                    tagName: state.tagName,
                    current: currentState.current,
                    after: state.after
                });
            }
            
            // 恢复到重做状态
            this.restoreElementState(state.element, state.after);
            
            this.log(`重做操作成功: ${state.tagName}`, 'success');
            
        } catch (error) {
            this.log(`重做操作失败: ${error.message}`, 'error');
        } finally {
            this.isPerformingUndoRedo = false;
        }
    }

    restoreElementState(element, state) {
        if (!element || !state) return;

        const tagName = element.tagName.toLowerCase();
        
        if (tagName === 'input' || tagName === 'textarea') {
            element.value = state.value;
            
            // 恢复光标位置
            if (typeof state.selectionStart === 'number' && typeof state.selectionEnd === 'number') {
                element.setSelectionRange(state.selectionStart, state.selectionEnd);
            }
        } else if (element.contentEditable === 'true') {
            element.innerHTML = state.innerHTML;
            
            // 恢复选择范围（简化版本）
            if (state.selection) {
                try {
                    const selection = window.getSelection();
                    selection.removeAllRanges();
                    
                    const range = document.createRange();
                    if (state.selection.startContainer && state.selection.endContainer) {
                        range.setStart(state.selection.startContainer, state.selection.startOffset);
                        range.setEnd(state.selection.endContainer, state.selection.endOffset);
                        selection.addRange(range);
                    }
                } catch (error) {
                    // 选择范围恢复失败，忽略
                    this.log('选择范围恢复失败', 'debug');
                }
            }
        }
    }
}

// 创建全局桥接客户端实例
const bridgeClient = new WebViewBridgeClient();

// 测试函数
function testConnection() {
    bridgeClient.sendMessage('testConnection', { 
        platform: bridgeClient.platform,
        userAgent: navigator.userAgent 
    });
}

function sendPing() {
    bridgeClient.sendMessage('ping', { 
        timestamp: Date.now() 
    }, (response) => {
        bridgeClient.log('收到Pong响应', 'success', response);
    });
}

function getSystemInfo() {
    bridgeClient.sendMessage('getSystemInfo', {}, (response) => {
        bridgeClient.log('系统信息', 'info', response);
    });
}

function sendCustomMessage() {
    const methodInput = document.getElementById('methodInput');
    const paramsInput = document.getElementById('paramsInput');
    
    if (!methodInput.value.trim()) {
        alert('请输入方法名');
        return;
    }
    
    let params = {};
    if (paramsInput.value.trim()) {
        try {
            params = JSON.parse(paramsInput.value);
        } catch (error) {
            alert('参数格式错误，请输入有效的JSON');
            return;
        }
    }
    
    bridgeClient.sendMessage(methodInput.value, params);
}

function clearInputs() {
    document.getElementById('methodInput').value = '';
    document.getElementById('paramsInput').value = '';
}

function saveData() {
    const testData = {
        id: Date.now(),
        name: '测试数据',
        value: Math.random(),
        timestamp: new Date().toISOString()
    };
    
    bridgeClient.sendMessage('saveData', { data: testData });
}

function loadData() {
    bridgeClient.sendMessage('loadData', {}, (response) => {
        bridgeClient.log('加载的数据', 'info', response);
    });
}

function deleteData() {
    if (confirm('确定要删除数据吗？')) {
        bridgeClient.sendMessage('deleteData', {});
    }
}

function clearLog() {
    bridgeClient.clearLog();
}

// 监听页面可见性变化，处理页面重新加载的情况
document.addEventListener('visibilitychange', function() {
    if (!document.hidden && window.bridgeClient) {
        // 页面重新可见时，检查桥接连接状态
        if (!window.bridgeClient.isConnected) {
            console.log('页面重新可见，重新初始化桥接连接');
            window.bridgeClient.init();
        }
    }
});

// 全局暴露bridgeClient
window.WebViewBridgeClient = bridgeClient;