#include <QApplication>
#include <QWidget>
#include <QVBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QDebug>
#include <QTimer>
#include <QMessageBox>

#ifdef Q_OS_WIN
#include "platform/windows/WindowsView.h"
#include "platform/windows/WindowsWebViewBridge.h"
#endif

/**
 * @brief 简单的WebView测试窗口
 * 用于验证WebView2是否能正常工作
 */
class WebViewTestWindow : public QWidget
{
    Q_OBJECT

public:
    WebViewTestWindow(QWidget* parent = nullptr) : QWidget(parent)
    {
        setupUI();
        setupWebView();
    }

private slots:
    void testConnection()
    {
        if (m_webView) {
            qDebug() << "Testing WebView connection...";
            m_statusLabel->setText("状态: 测试连接中...");
            
            // 发送测试消息
            QJsonObject params;
            params["test"] = "connection";
            m_webView->getWebViewBridge()->sendMessage("testMethod", params);
            
            // 3秒后检查状态
            QTimer::singleShot(3000, this, [this]() {
                m_statusLabel->setText("状态: 连接测试完成，请查看控制台输出");
            });
        }
    }
    
    void onWebViewReady()
    {
        qDebug() << "WebView is ready!";
        m_statusLabel->setText("状态: WebView已就绪 ✅");
        m_testButton->setEnabled(true);
        
        // 加载测试HTML
        QString testHtml = R"(
<!DOCTYPE html>
<html>
<head>
    <title>WebView2 连接测试</title>
    <style>
        body { 
            font-family: Arial, sans-serif; 
            padding: 20px; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-align: center;
        }
        .container {
            max-width: 600px;
            margin: 0 auto;
            background: rgba(255,255,255,0.1);
            padding: 30px;
            border-radius: 15px;
            backdrop-filter: blur(10px);
        }
        button {
            background: #4CAF50;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            margin: 10px;
        }
        button:hover { background: #45a049; }
        #status { 
            margin: 20px 0; 
            padding: 10px; 
            background: rgba(255,255,255,0.2); 
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🌐 WebView2 桥接测试</h1>
        <p>这是一个测试页面，用于验证Qt与WebView2的桥接功能</p>
        
        <div id="status">等待桥接连接...</div>
        
        <button onclick="testBridge()">测试桥接通信</button>
        <button onclick="sendPing()">发送Ping</button>
        <button onclick="getSystemInfo()">获取系统信息</button>
        
        <div id="messages" style="margin-top: 20px; text-align: left;"></div>
    </div>

    <script>
        // 等待桥接就绪
        function waitForBridge() {
            if (typeof window.qtBridge !== 'undefined') {
                document.getElementById('status').innerHTML = '✅ 桥接已连接';
                console.log('Qt Bridge is ready!');
            } else {
                setTimeout(waitForBridge, 100);
            }
        }
        waitForBridge();
        
        function addMessage(msg) {
            const div = document.createElement('div');
            div.style.cssText = 'background: rgba(255,255,255,0.1); padding: 5px; margin: 5px 0; border-radius: 3px; font-size: 14px;';
            div.textContent = new Date().toLocaleTimeString() + ': ' + msg;
            document.getElementById('messages').appendChild(div);
        }
        
        function testBridge() {
            if (window.qtBridge) {
                addMessage('发送测试消息到Qt...');
                window.qtBridge.sendMessage('testMethod', {message: 'Hello from WebView2!'});
            } else {
                addMessage('❌ 桥接未连接');
            }
        }
        
        function sendPing() {
            if (window.qtBridge) {
                addMessage('发送Ping...');
                window.qtBridge.sendMessage('ping', {timestamp: Date.now()});
            } else {
                addMessage('❌ 桥接未连接');
            }
        }
        
        function getSystemInfo() {
            if (window.qtBridge) {
                addMessage('请求系统信息...');
                window.qtBridge.sendMessage('getSystemInfo', {});
            } else {
                addMessage('❌ 桥接未连接');
            }
        }
        
        // 监听来自Qt的消息
        if (window.qtBridge) {
            window.qtBridge.onMessage = function(method, params) {
                addMessage('收到Qt消息: ' + method + ' - ' + JSON.stringify(params));
            };
        }
    </script>
</body>
</html>
        )";
        
        m_webView->loadHtml(testHtml);
    }

private:
    void setupUI()
    {
        setWindowTitle("WebView2 连接测试");
        setMinimumSize(800, 600);
        
        QVBoxLayout* layout = new QVBoxLayout(this);
        
        // 状态标签
        m_statusLabel = new QLabel("状态: 初始化中...");
        m_statusLabel->setStyleSheet("QLabel { font-size: 14px; padding: 10px; background: #f0f0f0; border-radius: 5px; }");
        layout->addWidget(m_statusLabel);
        
        // 测试按钮
        m_testButton = new QPushButton("测试桥接连接");
        m_testButton->setEnabled(false);
        connect(m_testButton, &QPushButton::clicked, this, &WebViewTestWindow::testConnection);
        layout->addWidget(m_testButton);
        
        setLayout(layout);
    }
    
    void setupWebView()
    {
#ifdef Q_OS_WIN
        qDebug() << "Setting up Windows WebView...";
        m_webView = new WindowsView(this);
        
        // 连接WebView就绪信号
        connect(m_webView, &WindowsView::viewReady, this, &WebViewTestWindow::onWebViewReady);
        
        // 将WebView添加到布局
        layout()->addWidget(m_webView);
        
        // 初始化WebView
        m_webView->initializeView();
        
        qDebug() << "Windows WebView setup completed";
#else
        m_statusLabel->setText("状态: 当前平台不支持WebView2 ❌");
        QMessageBox::warning(this, "警告", "当前平台不支持WebView2测试");
#endif
    }

private:
    QLabel* m_statusLabel;
    QPushButton* m_testButton;
    
#ifdef Q_OS_WIN
    WindowsView* m_webView;
#endif
};

int main(int argc, char *argv[])
{
    QApplication app(argc, argv);
    
    qDebug() << "Starting WebView2 connection test...";
    qDebug() << "Qt version:" << QT_VERSION_STR;
    
#ifdef Q_OS_WIN
    qDebug() << "Platform: Windows - WebView2 support enabled";
#else
    qDebug() << "Platform: Non-Windows - WebView2 not supported";
#endif
    
    WebViewTestWindow window;
    window.show();
    
    return app.exec();
}

#include "test_webview_simple.moc"