#ifndef WINDOWSWEBVIEWBRIDGE_H
#define WINDOWSWEBVIEWBRIDGE_H

#include "../../bridge/WebViewBridge.h"
#include <QObject>
#include <QWidget>
#include <QString>

#ifdef Q_OS_WIN

// 使用原生WebView2
#include <windows.h>
#include <WebView2.h>
#include <comdef.h>
#include <memory>
#include <functional>
#include <wrl.h>

/**
 * @brief Windows平台的WebView桥接实现
 * 使用原生WebView2控件与JavaScript进行通信
 */
class WindowsWebViewBridge : public WebViewBridge
{
    Q_OBJECT

public:
    explicit WindowsWebViewBridge(QWidget *parent = nullptr);
    ~WindowsWebViewBridge() override;

    // 实现基类的纯虚函数
    void sendMessage(const QString& method, const QJsonObject& params = QJsonObject()) override;
    void evaluateJavaScript(const QString& script) override;
    void initializeWebView() override;
    void loadHtml(const QString& html) override;
    void loadUrl(const QString& url) override;

    /**
     * @brief 调整WebView控件大小
     */
    void resizeWebView();

    /**
     * @brief 获取WebView2控制器
     */
    ICoreWebView2Controller* getController() const { return m_webViewController; }

private slots:
    void onParentResized();

private:
    /**
     * @brief 初始化COM组件
     */
    HRESULT initializeCOM();

    /**
     * @brief 清理COM组件
     */
    void cleanupCOM();

    /**
     * @brief 创建WebView2环境
     */
    HRESULT createWebViewEnvironment();

    /**
     * @brief 创建WebView2控制器
     */
    HRESULT createWebViewController();

    /**
     * @brief 设置WebView2事件处理
     */
    void setupEventHandlers();

    /**
     * @brief 注入桥接脚本
     */
    void injectBridgeScript();

    /**
     * @brief 向Web发送消息
     */
    void sendMessageToWeb(const QString& method, const QJsonObject& params);

    /**
     * @brief 处理来自Web的消息
     */
    void handleWebMessage(const QString& message);

private:
    QWidget* m_parentWidget;
    HWND m_parentHwnd;
    
    // WebView2 组件
    ICoreWebView2Environment* m_webViewEnvironment;
    ICoreWebView2Controller* m_webViewController;
    ICoreWebView2* m_webView;
    
    // 事件令牌
    EventRegistrationToken m_navigationCompletedToken;
    EventRegistrationToken m_webMessageReceivedToken;
    
    bool m_isInitialized;
    bool m_comInitialized;
    QString m_pendingUrl;
    QString m_pendingHtml;
};

#endif // Q_OS_WIN

#endif // WINDOWSWEBVIEWBRIDGE_H