#include "WindowsWebViewBridge.h"
#include <QDebug>
#include <QJsonDocument>
#include <QJsonObject>
#include <QApplication>
#include <QTimer>
#include <QDir>
#include <QStandardPaths>

#ifdef Q_OS_WIN

#if WEBVIEW_DEBUG_MODE
#include "../../debug/DebugManager.h"
#endif

WindowsWebViewBridge::WindowsWebViewBridge(QWidget* parent)
    : WebViewBridge(parent)
    , m_parentWidget(parent)
    , m_parentHwnd(nullptr)
    , m_webViewEnvironment(nullptr)
    , m_webViewController(nullptr)
    , m_webView(nullptr)
    , m_isInitialized(false)
    , m_comInitialized(false)
{
    qDebug() << "WindowsWebViewBridge::WindowsWebViewBridge() - Constructor called";
    qDebug() << "WindowsWebViewBridge::WindowsWebViewBridge() - Parent widget:" << parent;
    
    if (m_parentWidget) {
        m_parentHwnd = reinterpret_cast<HWND>(m_parentWidget->winId());
        qDebug() << "WindowsWebViewBridge::WindowsWebViewBridge() - Parent HWND:" << m_parentHwnd;
        
        // 连接父窗口的resize信号到我们的槽函数
        connect(m_parentWidget, QOverload<>::of(&QWidget::update), this, &WindowsWebViewBridge::onParentResized);
        qDebug() << "WindowsWebViewBridge::WindowsWebViewBridge() - Connected resize signal";
    } else {
        qWarning() << "WindowsWebViewBridge::WindowsWebViewBridge() - Parent widget is null!";
    }
    
    qDebug() << "WindowsWebViewBridge::WindowsWebViewBridge() - Constructor completed";
}

WindowsWebViewBridge::~WindowsWebViewBridge()
{
    qDebug() << "WindowsWebViewBridge destructor";
    
    if (m_webView) {
        // 取消注册事件
        if (m_navigationCompletedToken.value != 0) {
            m_webView->remove_NavigationCompleted(m_navigationCompletedToken);
        }
        if (m_webMessageReceivedToken.value != 0) {
            m_webView->remove_WebMessageReceived(m_webMessageReceivedToken);
        }
    }
    
    // 清理WebView2组件
    if (m_webView) {
        m_webView->Release();
        m_webView = nullptr;
    }
    if (m_webViewController) {
        m_webViewController->Release();
        m_webViewController = nullptr;
    }
    if (m_webViewEnvironment) {
        m_webViewEnvironment->Release();
        m_webViewEnvironment = nullptr;
    }
    
    cleanupCOM();
}

void WindowsWebViewBridge::initializeWebView()
{
    qDebug() << "WindowsWebViewBridge::initializeWebView() - Starting WebView2 initialization";
    qDebug() << "WindowsWebViewBridge::initializeWebView() - Current initialization state:" << m_isInitialized;
    
    if (m_isInitialized) {
        qWarning() << "WindowsWebViewBridge::initializeWebView() - WebView already initialized";
        return;
    }
    
    if (!m_parentHwnd) {
        qCritical() << "WindowsWebViewBridge::initializeWebView() - Parent HWND is null";
        return;
    }
    
    qDebug() << "WindowsWebViewBridge::initializeWebView() - Parent HWND valid:" << m_parentHwnd;
    
    // 初始化COM
    qDebug() << "WindowsWebViewBridge::initializeWebView() - Initializing COM";
    HRESULT hr = initializeCOM();
    if (FAILED(hr)) {
        qCritical() << "WindowsWebViewBridge::initializeWebView() - Failed to initialize COM:" << QString::number(hr, 16);
        return;
    }
    qDebug() << "WindowsWebViewBridge::initializeWebView() - COM initialized successfully";
    
    // 创建WebView2环境
    qDebug() << "WindowsWebViewBridge::initializeWebView() - Creating WebView2 environment";
    hr = createWebViewEnvironment();
    if (FAILED(hr)) {
        qCritical() << "WindowsWebViewBridge::initializeWebView() - Failed to create WebView2 environment:" << QString::number(hr, 16);
        return;
    }
    qDebug() << "WindowsWebViewBridge::initializeWebView() - WebView2 environment creation initiated";
}

HRESULT WindowsWebViewBridge::initializeCOM()
{
    qDebug() << "WindowsWebViewBridge::initializeCOM() - Checking COM initialization state:" << m_comInitialized;
    
    if (m_comInitialized) {
        qDebug() << "WindowsWebViewBridge::initializeCOM() - COM already initialized";
        return S_OK;
    }
    
    qDebug() << "WindowsWebViewBridge::initializeCOM() - Calling CoInitializeEx";
    HRESULT hr = CoInitializeEx(nullptr, COINIT_APARTMENTTHREADED);
    
    qDebug() << "WindowsWebViewBridge::initializeCOM() - CoInitializeEx result:" << QString::number(hr, 16);
    
    if (SUCCEEDED(hr) || hr == RPC_E_CHANGED_MODE) {
        m_comInitialized = true;
        qDebug() << "WindowsWebViewBridge::initializeCOM() - COM initialization successful";
        return S_OK;
    }
    
    qCritical() << "WindowsWebViewBridge::initializeCOM() - COM initialization failed with HRESULT:" << QString::number(hr, 16);
    return hr;
}

void WindowsWebViewBridge::cleanupCOM()
{
    if (m_comInitialized) {
        CoUninitialize();
        m_comInitialized = false;
    }
}

HRESULT WindowsWebViewBridge::createWebViewEnvironment()
{
    qDebug() << "WindowsWebViewBridge::createWebViewEnvironment() - Creating WebView2 environment";
    
    // 创建环境完成回调的简单实现
    class EnvironmentCallback : public ICoreWebView2CreateCoreWebView2EnvironmentCompletedHandler
    {
    private:
        WindowsWebViewBridge* m_bridge;
        ULONG m_refCount;
        
    public:
        EnvironmentCallback(WindowsWebViewBridge* bridge) : m_bridge(bridge), m_refCount(1) {
            qDebug() << "EnvironmentCallback::EnvironmentCallback() - Created";
        }
        
        HRESULT STDMETHODCALLTYPE QueryInterface(REFIID riid, void** ppvObject) override
        {
            if (riid == IID_IUnknown || riid == __uuidof(ICoreWebView2CreateCoreWebView2EnvironmentCompletedHandler))
            {
                *ppvObject = this;
                AddRef();
                return S_OK;
            }
            return E_NOINTERFACE;
        }
        
        ULONG STDMETHODCALLTYPE AddRef() override
        {
            return InterlockedIncrement(&m_refCount);
        }
        
        ULONG STDMETHODCALLTYPE Release() override
        {
            ULONG count = InterlockedDecrement(&m_refCount);
            if (count == 0)
                delete this;
            return count;
        }
        
        HRESULT STDMETHODCALLTYPE Invoke(HRESULT result, ICoreWebView2Environment* env) override
        {
            qDebug() << "EnvironmentCallback::Invoke() - Called with result:" << QString::number(result, 16);
            
            if (FAILED(result)) {
                qCritical() << "EnvironmentCallback::Invoke() - Failed to create WebView2 environment:" << QString::number(result, 16);
                return result;
            }
            
            qDebug() << "EnvironmentCallback::Invoke() - Environment created successfully";
            
            m_bridge->m_webViewEnvironment = env;
            if (m_bridge->m_webViewEnvironment) {
                m_bridge->m_webViewEnvironment->AddRef();
                qDebug() << "EnvironmentCallback::Invoke() - Environment reference added";
            }
            
            // 创建WebView控制器
            qDebug() << "EnvironmentCallback::Invoke() - Creating WebView controller";
            m_bridge->createWebViewController();
            return S_OK;
        }
    };
    
    // 在调试模式下，使用独立的用户数据目录，便于调试缓存与配置隔离
    std::wstring userDataFolderW;
#if WEBVIEW_DEBUG_MODE
    if (DebugManager::instance()->isDebugEnabled()) {
        const QString base = QStandardPaths::writableLocation(QStandardPaths::AppLocalDataLocation);
        const QString folder = QDir(base).filePath("WebView2DebugProfile");
        QDir().mkpath(folder);
        DebugManager::instance()->debugLog(QString("Using WebView2 userDataFolder: %1").arg(folder), "WindowsWebView");
        userDataFolderW = folder.toStdWString();
    }
#endif

    // 优先使用带选项的环境创建函数，以便传入 userDataFolder
    qDebug() << "WindowsWebViewBridge::createWebViewEnvironment() - Calling CreateCoreWebView2Environment[WithOptions]";
    HRESULT hr = S_OK;
    if (!userDataFolderW.empty()) {
        hr = CreateCoreWebView2EnvironmentWithOptions(
            /*browserExecutableFolder*/ nullptr,
            /*userDataFolder*/ userDataFolderW.c_str(),
            /*environmentOptions*/ nullptr,
            new EnvironmentCallback(this));
    } else {
        hr = CreateCoreWebView2Environment(new EnvironmentCallback(this));
    }
    qDebug() << "WindowsWebViewBridge::createWebViewEnvironment() - CreateCoreWebView2Environment result:" << QString::number(hr, 16);
    
    return hr;
}

HRESULT WindowsWebViewBridge::createWebViewController()
{
    if (!m_webViewEnvironment) {
        qWarning() << "WebView environment not initialized";
        return E_FAIL;
    }
    
    // 创建控制器完成回调的简单实现
    class ControllerCallback : public ICoreWebView2CreateCoreWebView2ControllerCompletedHandler
    {
    private:
        WindowsWebViewBridge* m_bridge;
        ULONG m_refCount;
        
    public:
        ControllerCallback(WindowsWebViewBridge* bridge) : m_bridge(bridge), m_refCount(1) {}
        
        HRESULT STDMETHODCALLTYPE QueryInterface(REFIID riid, void** ppvObject) override
        {
            if (riid == IID_IUnknown || riid == __uuidof(ICoreWebView2CreateCoreWebView2ControllerCompletedHandler))
            {
                *ppvObject = this;
                AddRef();
                return S_OK;
            }
            return E_NOINTERFACE;
        }
        
        ULONG STDMETHODCALLTYPE AddRef() override
        {
            return InterlockedIncrement(&m_refCount);
        }
        
        ULONG STDMETHODCALLTYPE Release() override
        {
            ULONG count = InterlockedDecrement(&m_refCount);
            if (count == 0)
                delete this;
            return count;
        }
        
        HRESULT STDMETHODCALLTYPE Invoke(HRESULT result, ICoreWebView2Controller* controller) override
        {
            if (FAILED(result)) {
                qCritical() << "Failed to create WebView2 controller:" << result;
                return result;
            }
            
            m_bridge->m_webViewController = controller;
            if (m_bridge->m_webViewController) {
                m_bridge->m_webViewController->AddRef();
            }
            
            // 获取WebView2核心接口
            HRESULT hr = m_bridge->m_webViewController->get_CoreWebView2(&m_bridge->m_webView);
            if (FAILED(hr)) {
                qCritical() << "Failed to get CoreWebView2:" << hr;
                return hr;
            }
            
            if (m_bridge->m_webView) {
                m_bridge->m_webView->AddRef();
            }

            // 在调试模式下启用开发者工具、右键菜单等设置
#if WEBVIEW_DEBUG_MODE
            if (DebugManager::instance()->isDebugEnabled() && m_bridge->m_webView) {
                ICoreWebView2Settings* settings = nullptr;
                if (SUCCEEDED(m_bridge->m_webView->get_Settings(&settings)) && settings) {
                    settings->put_AreDevToolsEnabled(TRUE);
                    settings->put_AreDefaultContextMenusEnabled(TRUE);
                    settings->put_IsScriptEnabled(TRUE);
                    settings->Release();
                }
                // 自动打开 DevTools 窗口，保持与 macOS 调试体验一致
                m_bridge->m_webView->OpenDevToolsWindow();
                DebugManager::instance()->debugLog("WebView2 DevTools enabled and opened", "WindowsWebView");
            }
#endif
            
            // 设置WebView大小
            m_bridge->resizeWebView();
            
            // 设置事件处理
            m_bridge->setupEventHandlers();
            
            m_bridge->m_isInitialized = true;
            
            // 如果有待加载的内容，现在加载
            if (!m_bridge->m_pendingUrl.isEmpty()) {
                m_bridge->loadUrl(m_bridge->m_pendingUrl);
                m_bridge->m_pendingUrl.clear();
            } else if (!m_bridge->m_pendingHtml.isEmpty()) {
                m_bridge->loadHtml(m_bridge->m_pendingHtml);
                m_bridge->m_pendingHtml.clear();
            }
            
            qDebug() << "WebView2 initialized successfully";
            return S_OK;
        }
    };
    
    return m_webViewEnvironment->CreateCoreWebView2Controller(
        m_parentHwnd,
        new ControllerCallback(this));
}

void WindowsWebViewBridge::setupEventHandlers()
{
    if (!m_webView) {
        return;
    }
    
    // 导航完成事件处理器
    class NavigationCompletedHandler : public ICoreWebView2NavigationCompletedEventHandler
    {
    private:
        WindowsWebViewBridge* m_bridge;
        ULONG m_refCount;
        
    public:
        NavigationCompletedHandler(WindowsWebViewBridge* bridge) : m_bridge(bridge), m_refCount(1) {}
        
        HRESULT STDMETHODCALLTYPE QueryInterface(REFIID riid, void** ppvObject) override
        {
            if (riid == IID_IUnknown || riid == __uuidof(ICoreWebView2NavigationCompletedEventHandler))
            {
                *ppvObject = this;
                AddRef();
                return S_OK;
            }
            return E_NOINTERFACE;
        }
        
        ULONG STDMETHODCALLTYPE AddRef() override
        {
            return InterlockedIncrement(&m_refCount);
        }
        
        ULONG STDMETHODCALLTYPE Release() override
        {
            ULONG count = InterlockedDecrement(&m_refCount);
            if (count == 0)
                delete this;
            return count;
        }
        
        HRESULT STDMETHODCALLTYPE Invoke(ICoreWebView2* sender, ICoreWebView2NavigationCompletedEventArgs* args) override
        {
            qDebug() << "WindowsWebViewBridge::NavigationCompletedHandler - Navigation completed";
            
            BOOL success = FALSE;
            args->get_IsSuccess(&success);
            
            if (success) {
                qDebug() << "WindowsWebViewBridge::NavigationCompletedHandler - Navigation successful, injecting bridge script";
                // 注入桥接脚本，稍微延迟确保页面完全加载
                QTimer::singleShot(200, m_bridge, &WindowsWebViewBridge::injectBridgeScript);
                // 发射准备就绪信号，保持与 macOS 行为一致
                QMetaObject::invokeMethod(m_bridge, "webViewReady", Qt::QueuedConnection);
            } else {
                qWarning() << "WindowsWebViewBridge::NavigationCompletedHandler - Navigation failed";
            }
            
            return S_OK;
        }
    };
    
    // Web消息接收事件处理器
    class WebMessageReceivedHandler : public ICoreWebView2WebMessageReceivedEventHandler
    {
    private:
        WindowsWebViewBridge* m_bridge;
        ULONG m_refCount;
        
    public:
        WebMessageReceivedHandler(WindowsWebViewBridge* bridge) : m_bridge(bridge), m_refCount(1) {}
        
        HRESULT STDMETHODCALLTYPE QueryInterface(REFIID riid, void** ppvObject) override
        {
            if (riid == IID_IUnknown || riid == __uuidof(ICoreWebView2WebMessageReceivedEventHandler))
            {
                *ppvObject = this;
                AddRef();
                return S_OK;
            }
            return E_NOINTERFACE;
        }
        
        ULONG STDMETHODCALLTYPE AddRef() override
        {
            return InterlockedIncrement(&m_refCount);
        }
        
        ULONG STDMETHODCALLTYPE Release() override
        {
            ULONG count = InterlockedDecrement(&m_refCount);
            if (count == 0)
                delete this;
            return count;
        }
        
        HRESULT STDMETHODCALLTYPE Invoke(ICoreWebView2* sender, ICoreWebView2WebMessageReceivedEventArgs* args) override
        {
            LPWSTR message;
            args->TryGetWebMessageAsString(&message);
            
            if (message) {
                QString qMessage = QString::fromWCharArray(message);
                m_bridge->handleWebMessage(qMessage);
                CoTaskMemFree(message);
            }
            
            return S_OK;
        }
    };
    
    // 添加导航完成事件处理
    m_webView->add_NavigationCompleted(new NavigationCompletedHandler(this), &m_navigationCompletedToken);
    
    // 添加Web消息接收事件处理
    m_webView->add_WebMessageReceived(new WebMessageReceivedHandler(this), &m_webMessageReceivedToken);
}

void WindowsWebViewBridge::loadHtml(const QString& html)
{
    if (!m_isInitialized || !m_webView) {
        m_pendingHtml = html;
        return;
    }
    
    std::wstring wHtml = html.toStdWString();
    m_webView->NavigateToString(wHtml.c_str());
}

void WindowsWebViewBridge::loadUrl(const QString& url)
{
    if (!m_isInitialized || !m_webView) {
        m_pendingUrl = url;
        return;
    }
    
    std::wstring wUrl = url.toStdWString();
    m_webView->Navigate(wUrl.c_str());
}

void WindowsWebViewBridge::sendMessage(const QString& method, const QJsonObject& params)
{
    qDebug() << "WindowsWebViewBridge::sendMessage() - Sending message:" << method << params;
    sendMessageToWeb(method, params);
}

void WindowsWebViewBridge::evaluateJavaScript(const QString& script)
{
    if (!m_isInitialized || !m_webView) {
        qWarning() << "WebView not initialized, cannot evaluate JavaScript";
        return;
    }
    
    std::wstring wScript = script.toStdWString();
    
    // JavaScript执行完成回调处理器
    class ExecuteScriptHandler : public ICoreWebView2ExecuteScriptCompletedHandler
    {
    private:
        ULONG m_refCount;
        
    public:
        ExecuteScriptHandler() : m_refCount(1) {}
        
        HRESULT STDMETHODCALLTYPE QueryInterface(REFIID riid, void** ppvObject) override
        {
            if (riid == IID_IUnknown || riid == __uuidof(ICoreWebView2ExecuteScriptCompletedHandler))
            {
                *ppvObject = this;
                AddRef();
                return S_OK;
            }
            return E_NOINTERFACE;
        }
        
        ULONG STDMETHODCALLTYPE AddRef() override
        {
            return InterlockedIncrement(&m_refCount);
        }
        
        ULONG STDMETHODCALLTYPE Release() override
        {
            ULONG count = InterlockedDecrement(&m_refCount);
            if (count == 0)
                delete this;
            return count;
        }
        
        HRESULT STDMETHODCALLTYPE Invoke(HRESULT errorCode, LPCWSTR resultObjectAsJson) override
        {
            if (FAILED(errorCode)) {
                qWarning() << "JavaScript execution failed:" << errorCode;
            } else if (resultObjectAsJson) {
                QString result = QString::fromWCharArray(resultObjectAsJson);
                qDebug() << "JavaScript result:" << result;
            }
            return S_OK;
        }
    };
    
    m_webView->ExecuteScript(wScript.c_str(), new ExecuteScriptHandler());
}

void WindowsWebViewBridge::resizeWebView()
{
    if (!m_webViewController || !m_parentWidget) {
        return;
    }
    
    RECT bounds;
    bounds.left = 0;
    bounds.top = 0;
    bounds.right = m_parentWidget->width();
    bounds.bottom = m_parentWidget->height();
    
    m_webViewController->put_Bounds(bounds);
}

void WindowsWebViewBridge::onParentResized()
{
    resizeWebView();
}

void WindowsWebViewBridge::injectBridgeScript()
{
    qDebug() << "WindowsWebViewBridge::injectBridgeScript() - Injecting bridge script";
    
    QString bridgeScript = R"(
        (function() {
            if (window.NativeBridge) {
                console.log('NativeBridge already exists');
                return; // 已经注入过了
            }
            
            console.log('Creating NativeBridge object');
            
            // 创建与bridge.js期望一致的NativeBridge对象
            window.NativeBridge = {
                sendMessage: function(method, params) {
                    console.log('NativeBridge.sendMessage called:', method, params);
                    var message = {
                        method: method,
                        params: params || {}
                    };
                    window.chrome.webview.postMessage(JSON.stringify(message));
                },
                
                onMessage: function(callback) {
                    console.log('NativeBridge.onMessage callback registered');
                    window._nativeBridgeCallback = callback;
                }
            };
            
            // 处理来自原生的消息
            window.handleNativeMessage = function(message) {
                console.log('handleNativeMessage called:', message);
                if (window._nativeBridgeCallback) {
                    window._nativeBridgeCallback(message);
                }
            };
            
            console.log('NativeBridge injected successfully');
            
            // 发送桥接准备就绪消息
            setTimeout(function() {
                if (window.NativeBridge) {
                    console.log('Sending bridgeReady message');
                    window.NativeBridge.sendMessage('bridgeReady', {});
                }
            }, 100);
        })();
    )";
    
    evaluateJavaScript(bridgeScript);
}

void WindowsWebViewBridge::sendMessageToWeb(const QString& method, const QJsonObject& params)
{
    qDebug() << "WindowsWebViewBridge::sendMessageToWeb() - Sending message to web:" << method << params;
    
    QJsonObject message;
    message["method"] = method;
    message["params"] = params;
    message["timestamp"] = QDateTime::currentMSecsSinceEpoch();
    
    QJsonDocument doc(message);
    QString jsonString = doc.toJson(QJsonDocument::Compact);
    
    QString script = QString("if (window.handleNativeMessage) { window.handleNativeMessage(%1); }").arg(jsonString);
    
    qDebug() << "WindowsWebViewBridge::sendMessageToWeb() - Executing script:" << script;
    evaluateJavaScript(script);
}

void WindowsWebViewBridge::handleWebMessage(const QString& message)
{
    qDebug() << "WindowsWebViewBridge::handleWebMessage() - Received web message:" << message;
    
    QJsonParseError error;
    QJsonDocument doc = QJsonDocument::fromJson(message.toUtf8(), &error);
    
    if (error.error != QJsonParseError::NoError) {
        qWarning() << "WindowsWebViewBridge::handleWebMessage() - Failed to parse web message:" << error.errorString();
        return;
    }
    
    QJsonObject obj = doc.object();
    QString method = obj["method"].toString();
    QJsonObject params = obj["params"].toObject();
    
    qDebug() << "WindowsWebViewBridge::handleWebMessage() - Method:" << method << "Params:" << params;
    
    // 处理特殊消息
    if (method == "bridgeReady") {
        qDebug() << "WindowsWebViewBridge::handleWebMessage() - Bridge ready message received";
        // 发送确认消息回前端
        sendMessageToWeb("bridgeReady", QJsonObject());
    }
    
    // 发射信号给上层处理
    emit messageReceived(method, params);
}

#endif // Q_OS_WIN