#include "MacOSWebViewBridge.h"

#ifdef Q_OS_MACOS

#import <WebKit/WebKit.h>
#import <Cocoa/Cocoa.h>
#include <QDebug>
#include <QJsonDocument>
#include <QVBoxLayout>

#if WEBVIEW_DEBUG_MODE
#include "../../debug/DebugManager.h"
#endif

// Objective-C委托类，处理WKWebView的消息
@interface MacOSWebViewDelegate : NSObject <WKScriptMessageHandler, WKNavigationDelegate>
@property (nonatomic, assign) MacOSWebViewBridge* bridge;
@end

@implementation MacOSWebViewDelegate

- (void)userContentController:(WKUserContentController *)userContentController 
      didReceiveScriptMessage:(WKScriptMessage *)message {
    if (self.bridge && [message.name isEqualToString:@"bridge"]) {
        NSString* messageBody = nil;
        
        if ([message.body isKindOfClass:[NSString class]]) {
            messageBody = (NSString*)message.body;
        } else if ([message.body isKindOfClass:[NSDictionary class]]) {
            NSError* error = nil;
            NSData* jsonData = [NSJSONSerialization dataWithJSONObject:message.body 
                                                              options:0 
                                                                error:&error];
            if (!error && jsonData) {
                messageBody = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
            }
        }
        
        if (messageBody) {
            QString qMessage = QString::fromNSString(messageBody);
            self.bridge->handleNativeMessage(qMessage);
        }
    }
}

- (void)webView:(WKWebView *)webView didFinishNavigation:(WKNavigation *)navigation {
    if (self.bridge) {
        // 页面加载完成，重新注入桥接脚本
        self.bridge->injectBridgeScript();
        
        // 发射准备就绪信号
        QMetaObject::invokeMethod(self.bridge, "webViewReady", Qt::QueuedConnection);
    }
}

@end

MacOSWebViewBridge::MacOSWebViewBridge(QWidget* parent)
    : WebViewBridge(parent)
    , m_webView(nullptr)
    , m_delegate(nullptr)
    , m_containerWidget(nullptr)
    , m_isInitialized(false)
{
    m_containerWidget = new QWidget(parent);
    setupWebView();
}

MacOSWebViewBridge::~MacOSWebViewBridge()
{
    if (m_delegate) {
        [m_delegate release];
    }
    if (m_webView) {
        [m_webView release];
    }
}

void MacOSWebViewBridge::setupWebView()
{
    // 创建WKWebView配置
    WKWebViewConfiguration* config = [[WKWebViewConfiguration alloc] init];
    
    // 创建用户内容控制器
    WKUserContentController* userContentController = [[WKUserContentController alloc] init];
    
    // 创建委托对象
    m_delegate = [[MacOSWebViewDelegate alloc] init];
    m_delegate.bridge = this;
    
    // 添加消息处理器
    [userContentController addScriptMessageHandler:m_delegate name:@"bridge"];
    
    config.userContentController = userContentController;
    
#if WEBVIEW_DEBUG_MODE
    // 启用Safari开发者工具支持（仅在调试模式下）
    if (DebugManager::instance()->shouldEnableSafariDevTools()) {
        // 启用开发者菜单
        [config.preferences setValue:@YES forKey:@"developerExtrasEnabled"];
        
        // 启用Web Inspector - 这是关键设置
        if (@available(macOS 10.15, *)) {
            config.preferences.javaScriptCanOpenWindowsAutomatically = YES;
        }
        
        // 启用右键菜单中的"检查元素"
        if ([config respondsToSelector:@selector(setInspectable:)]) {
            [config setValue:@YES forKey:@"inspectable"];
        }
        
        DebugManager::instance()->debugLog("Safari Developer Tools enabled with remote debugging", "WebView");
    } else {
        DebugManager::instance()->debugLog("Safari Developer Tools disabled by runtime setting", "WebView");
    }
#else
    // 在非调试模式下，确保开发者工具完全禁用
    [config.preferences setValue:@NO forKey:@"developerExtrasEnabled"];
#endif
    
    // 创建WKWebView
    NSRect frame = NSMakeRect(0, 0, 800, 600);
    m_webView = [[WKWebView alloc] initWithFrame:frame configuration:config];
    m_webView.navigationDelegate = m_delegate;
    
    // 配置WebView以支持键盘事件
    // 注意：setAcceptsTouchBar 和 setCanBecomeKeyView 在 WKWebView 中不可用
    // WKWebView 继承自 NSView，默认已经支持键盘事件
    
    // 确保WebView可以接收键盘事件
    if ([m_webView respondsToSelector:@selector(setAcceptsFirstResponder:)]) {
        [m_webView setValue:@YES forKey:@"acceptsFirstResponder"];
    }
    
#if WEBVIEW_DEBUG_MODE
    // 在调试模式下启用额外的调试功能
    if (DebugManager::instance()->shouldEnableSafariDevTools()) {
        // 启用右键菜单
        if ([m_webView respondsToSelector:@selector(setAllowsBackForwardNavigationGestures:)]) {
            [m_webView setAllowsBackForwardNavigationGestures:YES];
        }
        
        // 确保WebView可以被远程调试 - 使用正确的API
        if (@available(macOS 13.3, *)) {
            if ([m_webView respondsToSelector:@selector(setInspectable:)]) {
                [m_webView setInspectable:YES];
            }
        }
        
        DebugManager::instance()->debugLog("WebView debug features and remote debugging enabled", "WebView");
    }
#endif
    
    // 将WKWebView嵌入到Qt Widget中
    if (m_containerWidget) {
        NSView* parentView = (__bridge NSView*)m_containerWidget->winId();
        if (parentView) {
            [parentView addSubview:m_webView];
            
            // 设置自动布局
            m_webView.translatesAutoresizingMaskIntoConstraints = NO;
            [NSLayoutConstraint activateConstraints:@[
                [m_webView.topAnchor constraintEqualToAnchor:parentView.topAnchor],
                [m_webView.bottomAnchor constraintEqualToAnchor:parentView.bottomAnchor],
                [m_webView.leadingAnchor constraintEqualToAnchor:parentView.leadingAnchor],
                [m_webView.trailingAnchor constraintEqualToAnchor:parentView.trailingAnchor]
            ]];
            
            // 确保WebView可以成为第一响应者并接收键盘事件
            dispatch_async(dispatch_get_main_queue(), ^{
                [[m_webView window] makeFirstResponder:m_webView];
                [m_webView becomeFirstResponder];
            });
        }
    }
    
    [config release];
    [userContentController release];
}

void MacOSWebViewBridge::initializeWebView()
{
    if (m_isInitialized) {
        return;
    }
    
    injectBridgeScript();
    m_isInitialized = true;
}

void MacOSWebViewBridge::injectBridgeScript()
{
    // 注入桥接JavaScript代码
    QString bridgeScript = R"(
        window.NativeBridge = {
            sendMessage: function(method, params) {
                var message = {
                    method: method,
                    params: params || {}
                };
                window.webkit.messageHandlers.bridge.postMessage(JSON.stringify(message));
            },
            
            // 用于接收来自Native的消息
            onMessage: function(callback) {
                window.nativeMessageCallback = callback;
            }
        };
        
        // 通知Native桥接已准备就绪
        window.NativeBridge.sendMessage('bridgeReady', {});
    )";
    
    evaluateJavaScript(bridgeScript);
}

void MacOSWebViewBridge::sendMessage(const QString& method, const QJsonObject& params)
{
    QJsonObject message;
    message["method"] = method;
    message["params"] = params;
    
    QJsonDocument doc(message);
    QString jsonString = doc.toJson(QJsonDocument::Compact);
    
    // 调用JavaScript回调函数
    QString script = QString("if (window.nativeMessageCallback) { window.nativeMessageCallback(%1); }")
                        .arg(jsonString);
    
    evaluateJavaScript(script);
}

void MacOSWebViewBridge::evaluateJavaScript(const QString& script)
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    NSString* nsScript = script.toNSString();
    [m_webView evaluateJavaScript:nsScript completionHandler:^(id result, NSError* error) {
        if (error) {
            qWarning() << "JavaScript execution error:" << QString::fromNSString(error.localizedDescription);
        }
    }];
}

void MacOSWebViewBridge::loadHtml(const QString& html)
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    // 存储HTML内容，用于reload
    m_htmlContent = html;
    
    NSString* nsHtml = html.toNSString();
    
    // 使用about:blank作为baseURL，这是最安全的方式
    NSURL* baseURL = [NSURL URLWithString:@"about:blank"];
    
#if WEBVIEW_DEBUG_MODE
    DebugManager::instance()->debugLog(QString("Loading HTML with baseURL: about:blank"), "WebView");
#endif
    
    [m_webView loadHTMLString:nsHtml baseURL:baseURL];
}

void MacOSWebViewBridge::loadUrl(const QString& url)
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    NSURL* nsUrl = [NSURL URLWithString:url.toNSString()];
    NSURLRequest* request = [NSURLRequest requestWithURL:nsUrl];
    [m_webView loadRequest:request];
}

QWidget* MacOSWebViewBridge::getWebViewWidget()
{
    return m_containerWidget;
}

void MacOSWebViewBridge::handleNativeMessage(const QString& message)
{
    // 调用基类的消息处理方法
    handleMessage(message);
}

void MacOSWebViewBridge::sendKeyboardEvent(int key, Qt::KeyboardModifiers modifiers, const QString& eventType)
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    // 将Qt键码转换为JavaScript键码
    QString keyName;
    QString keyCode;
    
    switch (key) {
        case Qt::Key_Z:
            keyName = "z";
            keyCode = "KeyZ";
            break;
        case Qt::Key_X:
            keyName = "x";
            keyCode = "KeyX";
            break;
        case Qt::Key_C:
            keyName = "c";
            keyCode = "KeyC";
            break;
        case Qt::Key_V:
            keyName = "v";
            keyCode = "KeyV";
            break;
        case Qt::Key_A:
            keyName = "a";
            keyCode = "KeyA";
            break;
        default:
            qWarning() << "Unsupported key for keyboard event:" << key;
            return;
    }
    
    // 直接调用JavaScript的handleKeyboardEvent方法
    QString script = QString(R"(
        (function() {
            try {
                // 等待WebViewBridgeClient加载完成
                var maxRetries = 10;
                var retryCount = 0;
                
                function tryCallHandler() {
                    console.log('MacOSWebViewBridge: Attempting to call handleKeyboardEvent, retry:', retryCount);
                    console.log('MacOSWebViewBridge: window.WebViewBridgeClient exists:', !!window.WebViewBridgeClient);
                    
                    if (window.WebViewBridgeClient && typeof window.WebViewBridgeClient.handleKeyboardEvent === 'function') {
                        // 创建键盘事件对象
                        var event = new KeyboardEvent('%1', {
                            key: '%2',
                            code: '%3',
                            ctrlKey: %4,
                            altKey: %5,
                            shiftKey: %6,
                            metaKey: %7,
                            bubbles: true,
                            cancelable: true
                        });
                        
                        // 获取当前焦点元素
                        var activeElement = document.activeElement;
                        var target = activeElement;
                        
                        // 如果没有焦点元素或焦点在body上，尝试找到可编辑元素
                        if (!target || target === document.body || target === document.documentElement) {
                            var editableElements = document.querySelectorAll('input, textarea, [contenteditable="true"]');
                            if (editableElements.length > 0) {
                                target = editableElements[0];
                                target.focus();
                            } else {
                                target = document;
                            }
                        }
                        
                        // 设置事件的target属性
                        Object.defineProperty(event, 'target', {
                            value: target,
                            enumerable: true
                        });
                        
                        console.log('MacOSWebViewBridge: Successfully calling handleKeyboardEvent', {
                            eventType: '%1',
                            key: '%2',
                            target: target.tagName || 'document'
                        });
                        
                        // 调用处理方法
                        window.WebViewBridgeClient.handleKeyboardEvent(event);
                        return true;
                    } else {
                        retryCount++;
                        if (retryCount < maxRetries) {
                            console.log('MacOSWebViewBridge: WebViewBridgeClient not ready, retrying in 50ms...');
                            setTimeout(tryCallHandler, 50);
                            return false;
                        } else {
                            console.error('MacOSWebViewBridge: WebViewBridgeClient or handleKeyboardEvent method not found after', maxRetries, 'retries');
                            console.log('MacOSWebViewBridge: Available window properties:', Object.keys(window));
                            return false;
                        }
                    }
                }
                
                return tryCallHandler();
            } catch (error) {
                console.error('MacOSWebViewBridge: Error calling handleKeyboardEvent:', error);
                return false;
            }
        })();
    )")
    .arg(eventType)
    .arg(keyName)
    .arg(keyCode)
    .arg(modifiers & Qt::ControlModifier ? "true" : "false")
    .arg(modifiers & Qt::AltModifier ? "true" : "false")
    .arg(modifiers & Qt::ShiftModifier ? "true" : "false")
    .arg(modifiers & Qt::MetaModifier ? "true" : "false");
    
    qDebug() << "MacOSWebViewBridge: Sending keyboard event:" << eventType << keyName;
    evaluateJavaScript(script);
}

void MacOSWebViewBridge::setWebViewFocused(bool focused)
{
    if (!m_webView) {
        return;
    }
    
    if (focused) {
        // 设置WebView为第一响应者
        dispatch_async(dispatch_get_main_queue(), ^{
            NSWindow* window = [m_webView window];
            if (window) {
                [window makeFirstResponder:m_webView];
                [m_webView becomeFirstResponder];
            }
        });
        
        // 通过JavaScript聚焦到可编辑元素
        QString script = R"(
            (function() {
                try {
                    // 尝试聚焦到第一个可编辑元素
                    var editableElements = document.querySelectorAll('input, textarea, [contenteditable="true"]');
                    if (editableElements.length > 0) {
                        editableElements[0].focus();
                        console.log('MacOSWebViewBridge: Focused on editable element:', editableElements[0].tagName);
                    } else {
                        // 如果没有可编辑元素，聚焦到body
                        if (document.body) {
                            document.body.focus();
                            console.log('MacOSWebViewBridge: Focused on body');
                        }
                    }
                } catch (error) {
                    console.error('MacOSWebViewBridge: Error setting focus:', error);
                }
            })();
        )";
        evaluateJavaScript(script);
    } else {
        // 移除焦点
        QString script = R"(
            (function() {
                try {
                    if (document.activeElement && document.activeElement.blur) {
                        document.activeElement.blur();
                        console.log('MacOSWebViewBridge: Removed focus from element');
                    }
                } catch (error) {
                    console.error('MacOSWebViewBridge: Error removing focus:', error);
                }
            })();
        )";
        evaluateJavaScript(script);
    }
}

bool MacOSWebViewBridge::isWebViewFocused() const
{
    if (!m_webView) {
        return false;
    }
    
    // 检查WKWebView是否是当前的第一响应者
    NSWindow* window = [m_webView window];
    if (window) {
        NSResponder* firstResponder = [window firstResponder];
        return (firstResponder == m_webView) || [firstResponder isKindOfClass:[WKWebView class]];
    }
    
    return false;
}

void MacOSWebViewBridge::reloadWebView()
{
    if (!m_webView) {
        qWarning() << "MacOSWebViewBridge: Cannot reload - WebView not initialized";
        return;
    }
    
    if (m_htmlContent.isEmpty()) {
        qWarning() << "MacOSWebViewBridge: Cannot reload - No HTML content stored";
        return;
    }
    
    qDebug() << "MacOSWebViewBridge: Reloading WebView with stored HTML content";
    
    // 重新加载存储的HTML内容，而不是调用系统的reload
    NSString* nsHtml = m_htmlContent.toNSString();
    NSURL* baseURL = [NSURL URLWithString:@"about:blank"];
    
#if WEBVIEW_DEBUG_MODE
    DebugManager::instance()->debugLog(QString("Reloading HTML with baseURL: about:blank"), "WebView");
#endif
    
    [m_webView loadHTMLString:nsHtml baseURL:baseURL];
}

#endif // Q_OS_MACOS