#ifndef MACOSWEBVIEWBRIDGE_H
#define MACOSWEBVIEWBRIDGE_H

#include "../../bridge/WebViewBridge.h"

#ifdef Q_OS_MACOS

#include <QWidget>

// 前向声明Objective-C类
#ifdef __OBJC__
@class WKWebView;
@class MacOSWebViewDelegate;
#else
class WKWebView;
class MacOSWebViewDelegate;
#endif

/**
 * @brief macOS平台的WKWebView桥接实现
 * 使用WKWebView和window.webkit.messageHandlers进行通信
 */
class MacOSWebViewBridge : public WebViewBridge
{
    Q_OBJECT

public:
    explicit MacOSWebViewBridge(QWidget* parent = nullptr);
    ~MacOSWebViewBridge() override;

    // 实现基类的纯虚函数
    void sendMessage(const QString& method, const QJsonObject& params = QJsonObject()) override;
    void evaluateJavaScript(const QString& script) override;
    void initializeWebView() override;
    void loadHtml(const QString& html) override;
    void loadUrl(const QString& url) override;

    /**
     * @brief 获取WebView的原生视图
     * @return QWidget指针，可以添加到Qt布局中
     */
    QWidget* getWebViewWidget();

    /**
     * @brief 处理从WKWebView接收到的消息（由Objective-C代码调用）
     * @param message 消息内容
     */
    void handleNativeMessage(const QString& message);

    /**
     * @brief 向WebView发送键盘事件
     * @param key 按键代码
     * @param modifiers 修饰键
     * @param eventType 事件类型（keydown/keyup）
     */
    void sendKeyboardEvent(int key, Qt::KeyboardModifiers modifiers, const QString& eventType = "keydown");

    /**
     * @brief 设置WebView的焦点状态
     * @param focused 是否获得焦点
     */
    void setWebViewFocused(bool focused);

    /**
     * @brief 检查WebView是否有焦点
     * @return 如果WebView有焦点返回true
     */
    bool isWebViewFocused() const;

    /**
     * @brief 重新加载WebView内容
     */
    void reloadWebView();

    /**
     * @brief 注入桥接脚本（供delegate调用）
     */
    void injectBridgeScript();

private:
    void setupWebView();

private:
    WKWebView* m_webView;
    MacOSWebViewDelegate* m_delegate;
    QWidget* m_containerWidget;
    bool m_isInitialized;
    QString m_htmlContent; // 存储当前加载的HTML内容，用于reload
};

#endif // Q_OS_MACOS

#endif // MACOSWEBVIEWBRIDGE_H