#include "MacOSView.h"

#ifdef Q_OS_MACOS

#include "MacOSWebViewBridge.h"
#include "MacOSMenu.h"
#include <QVBoxLayout>
#include <QDebug>
#include <QApplication>

MacOSView::MacOSView(QWidget *parent)
    : BaseView(parent)
    , m_webViewBridge(nullptr)
    , m_menu(nullptr)
{
    m_webViewBridge = new MacOSWebViewBridge(this);
    setupLayout();
    setupMenu();
    setupKeyboardEventHandling();
    
    // 设置窗口属性
    setWindowTitle("WebView Bridge Demo");
    resize(1000, 700);
    setMinimumSize(800, 600);
    
    // 连接WebView准备就绪信号
    connect(m_webViewBridge, &WebViewBridge::webViewReady,
            this, &BaseView::viewReady);
}

MacOSView::~MacOSView()
{
    // m_webViewBridge和m_menu会被Qt的父子关系自动删除
}

WebViewBridge* MacOSView::getWebViewBridge()
{
    return m_webViewBridge;
}

void MacOSView::initializeView()
{
    if (m_webViewBridge) {
        m_webViewBridge->initializeWebView();
    }
}

void MacOSView::loadHtml(const QString& html)
{
    if (m_webViewBridge) {
        m_webViewBridge->loadHtml(html);
    }
}

void MacOSView::loadUrl(const QString& url)
{
    if (m_webViewBridge) {
        m_webViewBridge->loadUrl(url);
    }
}

void MacOSView::showView()
{
    // 强制显示窗口
    setWindowFlags(Qt::Window | Qt::WindowTitleHint | Qt::WindowCloseButtonHint | Qt::WindowMinMaxButtonsHint);
    show();
    raise();
    activateWindow();
    
    // 确保窗口显示在前台
    setAttribute(Qt::WA_ShowWithoutActivating, false);
    setWindowState(windowState() & ~Qt::WindowMinimized);
    
    // 强制将窗口置于最前面
    setWindowState(Qt::WindowActive);
    
    qDebug() << "MacOSView window shown with size:" << size();
    qDebug() << "Window flags:" << windowFlags();
    qDebug() << "Window state:" << windowState();
}

void MacOSView::setupLayout()
{
    QVBoxLayout* layout = new QVBoxLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    
    if (m_webViewBridge) {
        QWidget* webViewWidget = m_webViewBridge->getWebViewWidget();
        if (webViewWidget) {
            layout->addWidget(webViewWidget);
        }
    }
    
    setLayout(layout);
}

void MacOSView::setupMenu()
{
    m_menu = new MacOSMenu(this);
    if (m_menu && m_webViewBridge) {
        m_menu->setWebViewBridge(m_webViewBridge);
    }
}

void MacOSView::setupKeyboardEventHandling()
{
    // 安装事件过滤器来捕获键盘事件
    installEventFilter(this);
    
    // 设置焦点策略，确保窗口可以接收键盘事件
    setFocusPolicy(Qt::StrongFocus);
    
    // 确保WebView也能接收键盘事件
    if (m_webViewBridge) {
        QWidget* webViewWidget = m_webViewBridge->getWebViewWidget();
        if (webViewWidget) {
            webViewWidget->setFocusPolicy(Qt::StrongFocus);
            webViewWidget->installEventFilter(this);
        }
    }
}

void MacOSView::keyPressEvent(QKeyEvent* event)
{
    qDebug() << "MacOSView::keyPressEvent - Key:" << event->key() << "Modifiers:" << event->modifiers();
    
    // 处理快捷键
    if (handleShortcut(event->key(), event->modifiers())) {
        event->accept();
        return;
    }
    
    // 如果不是我们处理的快捷键，传递给基类
    BaseView::keyPressEvent(event);
}

void MacOSView::keyReleaseEvent(QKeyEvent* event)
{
    qDebug() << "MacOSView::keyReleaseEvent - Key:" << event->key() << "Modifiers:" << event->modifiers();
    BaseView::keyReleaseEvent(event);
}

bool MacOSView::eventFilter(QObject* obj, QEvent* event)
{
    if (event->type() == QEvent::KeyPress) {
        QKeyEvent* keyEvent = static_cast<QKeyEvent*>(event);
        qDebug() << "MacOSView::eventFilter - KeyPress - Key:" << keyEvent->key() << "Modifiers:" << keyEvent->modifiers();
        
        // 处理快捷键
        if (handleShortcut(keyEvent->key(), keyEvent->modifiers())) {
            return true; // 事件已处理
        }
    }
    
    // 传递给基类处理
    return BaseView::eventFilter(obj, event);
}

bool MacOSView::handleShortcut(int key, Qt::KeyboardModifiers modifiers)
{
    // 检查是否是Cmd键组合（在macOS上是MetaModifier）
    if (!(modifiers & Qt::MetaModifier)) {
        return false;
    }
    
    bool handled = false;
    
    switch (key) {
        case Qt::Key_Z:
            if (modifiers & Qt::ShiftModifier) {
                // Cmd+Shift+Z - 重做
                qDebug() << "MacOSView: Handling Cmd+Shift+Z (Redo)";
                if (m_menu) {
                    // 直接调用菜单的重做函数
                    QMetaObject::invokeMethod(m_menu, "onRedo", Qt::QueuedConnection);
                }
            } else {
                // Cmd+Z - 撤销
                qDebug() << "MacOSView: Handling Cmd+Z (Undo)";
                if (m_menu) {
                    QMetaObject::invokeMethod(m_menu, "onUndo", Qt::QueuedConnection);
                }
            }
            handled = true;
            break;
            
        case Qt::Key_X:
            // Cmd+X - 剪切
            qDebug() << "MacOSView: Handling Cmd+X (Cut)";
            if (m_menu) {
                QMetaObject::invokeMethod(m_menu, "onCut", Qt::QueuedConnection);
            }
            handled = true;
            break;
            
        case Qt::Key_C:
            // Cmd+C - 复制
            qDebug() << "MacOSView: Handling Cmd+C (Copy)";
            if (m_menu) {
                QMetaObject::invokeMethod(m_menu, "onCopy", Qt::QueuedConnection);
            }
            handled = true;
            break;
            
        case Qt::Key_V:
            // Cmd+V - 粘贴
            qDebug() << "MacOSView: Handling Cmd+V (Paste)";
            if (m_menu) {
                QMetaObject::invokeMethod(m_menu, "onPaste", Qt::QueuedConnection);
            }
            handled = true;
            break;
            
        case Qt::Key_A:
            // Cmd+A - 全选
            qDebug() << "MacOSView: Handling Cmd+A (Select All)";
            if (m_menu) {
                QMetaObject::invokeMethod(m_menu, "onSelectAll", Qt::QueuedConnection);
            }
            handled = true;
            break;
            
        default:
            // 不是我们处理的快捷键
            break;
    }
    
    return handled;
}

#endif // Q_OS_MACOS