#include "MacOSMenu.h"

#ifdef Q_OS_MACOS

#include "MacOSWebViewBridge.h"
#include <QApplication>
#include <QKeyEvent>
#include <QDebug>
#include <QTimer>

MacOSMenu::MacOSMenu(QObject *parent)
    : QObject(parent)
    , m_menuBar(nullptr)
    , m_webViewBridge(nullptr)
    , m_fileMenu(nullptr)
    , m_editMenu(nullptr)
    , m_viewMenu(nullptr)
    , m_windowMenu(nullptr)
    , m_helpMenu(nullptr)
{
    createNativeMenuBar();
}

MacOSMenu::~MacOSMenu()
{
    // Qt会自动清理菜单和动作
}

void MacOSMenu::createNativeMenuBar()
{
    m_menuBar = new QMenuBar(nullptr);
    
    createFileMenu();
    createEditMenu();
    createViewMenu();
    createWindowMenu();
    createHelpMenu();
}

void MacOSMenu::setWebViewBridge(MacOSWebViewBridge* bridge)
{
    m_webViewBridge = bridge;
}

void MacOSMenu::createFileMenu()
{
    m_fileMenu = m_menuBar->addMenu(tr("File"));
    
    m_newAction = m_fileMenu->addAction(tr("New"));
    m_newAction->setShortcut(QKeySequence::New);
    connect(m_newAction, &QAction::triggered, this, &MacOSMenu::onNewFile);
    
    m_openAction = m_fileMenu->addAction(tr("Open..."));
    m_openAction->setShortcut(QKeySequence::Open);
    connect(m_openAction, &QAction::triggered, this, &MacOSMenu::onOpenFile);
    
    m_fileMenu->addSeparator();
    
    m_saveAction = m_fileMenu->addAction(tr("Save"));
    m_saveAction->setShortcut(QKeySequence::Save);
    connect(m_saveAction, &QAction::triggered, this, &MacOSMenu::onSaveFile);
    
    m_saveAsAction = m_fileMenu->addAction(tr("Save As..."));
    m_saveAsAction->setShortcut(QKeySequence::SaveAs);
    connect(m_saveAsAction, &QAction::triggered, this, &MacOSMenu::onSaveAsFile);
    
    m_fileMenu->addSeparator();
    
    m_quitAction = m_fileMenu->addAction(tr("Quit"));
    m_quitAction->setShortcut(QKeySequence::Quit);
    connect(m_quitAction, &QAction::triggered, this, &MacOSMenu::onQuit);
}

void MacOSMenu::createEditMenu()
{
    m_editMenu = m_menuBar->addMenu(tr("Edit"));
    
    // 撤销 - Cmd+Z
    m_undoAction = m_editMenu->addAction(tr("Undo"));
    m_undoAction->setShortcut(QKeySequence::Undo);
    connect(m_undoAction, &QAction::triggered, this, &MacOSMenu::onUndo);
    
    // 重做 - Cmd+Shift+Z
    m_redoAction = m_editMenu->addAction(tr("Redo"));
    m_redoAction->setShortcut(QKeySequence::Redo);
    connect(m_redoAction, &QAction::triggered, this, &MacOSMenu::onRedo);
    
    m_editMenu->addSeparator();
    
    // 剪切 - Cmd+X
    m_cutAction = m_editMenu->addAction(tr("Cut"));
    m_cutAction->setShortcut(QKeySequence::Cut);
    connect(m_cutAction, &QAction::triggered, this, &MacOSMenu::onCut);
    
    // 复制 - Cmd+C
    m_copyAction = m_editMenu->addAction(tr("Copy"));
    m_copyAction->setShortcut(QKeySequence::Copy);
    connect(m_copyAction, &QAction::triggered, this, &MacOSMenu::onCopy);
    
    // 粘贴 - Cmd+V
    m_pasteAction = m_editMenu->addAction(tr("Paste"));
    m_pasteAction->setShortcut(QKeySequence::Paste);
    connect(m_pasteAction, &QAction::triggered, this, &MacOSMenu::onPaste);
    
    m_editMenu->addSeparator();
    
    // 全选 - Cmd+A
    m_selectAllAction = m_editMenu->addAction(tr("Select All"));
    m_selectAllAction->setShortcut(QKeySequence::SelectAll);
    connect(m_selectAllAction, &QAction::triggered, this, &MacOSMenu::onSelectAll);
}

void MacOSMenu::createViewMenu()
{
    m_viewMenu = m_menuBar->addMenu(tr("View"));
    
    // 重新加载 - Cmd+R
    m_reloadAction = m_viewMenu->addAction(tr("Reload"));
    m_reloadAction->setShortcut(QKeySequence::Refresh);
    connect(m_reloadAction, &QAction::triggered, this, &MacOSMenu::onReload);
    
    m_viewMenu->addSeparator();
    
    m_fullScreenAction = m_viewMenu->addAction(tr("Enter Full Screen"));
    m_fullScreenAction->setShortcut(QKeySequence("Ctrl+Cmd+F"));
    connect(m_fullScreenAction, &QAction::triggered, this, &MacOSMenu::onToggleFullScreen);
    
    m_viewMenu->addSeparator();
    
    m_zoomInAction = m_viewMenu->addAction(tr("Zoom In"));
    m_zoomInAction->setShortcut(QKeySequence::ZoomIn);
    connect(m_zoomInAction, &QAction::triggered, this, &MacOSMenu::onZoomIn);
    
    m_zoomOutAction = m_viewMenu->addAction(tr("Zoom Out"));
    m_zoomOutAction->setShortcut(QKeySequence::ZoomOut);
    connect(m_zoomOutAction, &QAction::triggered, this, &MacOSMenu::onZoomOut);
    
    m_actualSizeAction = m_viewMenu->addAction(tr("Actual Size"));
    m_actualSizeAction->setShortcut(QKeySequence("Cmd+0"));
    connect(m_actualSizeAction, &QAction::triggered, this, &MacOSMenu::onActualSize);
}

void MacOSMenu::createWindowMenu()
{
    m_windowMenu = m_menuBar->addMenu(tr("Window"));
    
    m_minimizeAction = m_windowMenu->addAction(tr("Minimize"));
    m_minimizeAction->setShortcut(QKeySequence("Cmd+M"));
    connect(m_minimizeAction, &QAction::triggered, this, &MacOSMenu::onMinimize);
    
    m_zoomAction = m_windowMenu->addAction(tr("Zoom"));
    connect(m_zoomAction, &QAction::triggered, this, &MacOSMenu::onZoom);
    
    m_windowMenu->addSeparator();
    
    m_bringAllToFrontAction = m_windowMenu->addAction(tr("Bring All to Front"));
    connect(m_bringAllToFrontAction, &QAction::triggered, this, &MacOSMenu::onBringAllToFront);
}

void MacOSMenu::createHelpMenu()
{
    m_helpMenu = m_menuBar->addMenu(tr("Help"));
    
    m_helpAction = m_helpMenu->addAction(tr("Help"));
    connect(m_helpAction, &QAction::triggered, this, &MacOSMenu::onHelp);
    
    m_aboutAction = m_helpMenu->addAction(tr("About"));
    connect(m_aboutAction, &QAction::triggered, this, &MacOSMenu::onAbout);
}

void MacOSMenu::forwardKeyEventToWebView(int key, Qt::KeyboardModifiers modifiers)
{
    if (!m_webViewBridge) {
        qDebug() << "MacOSMenu: WebView bridge not set, cannot forward key event";
        return;
    }
    
    // 使用新的sendKeyboardEvent方法
    MacOSWebViewBridge* macOSBridge = static_cast<MacOSWebViewBridge*>(m_webViewBridge);
    if (macOSBridge) {
        // 发送keydown事件
        macOSBridge->sendKeyboardEvent(key, modifiers, "keydown");
        
        // 稍后发送keyup事件
        QTimer::singleShot(50, [macOSBridge, key, modifiers]() {
            macOSBridge->sendKeyboardEvent(key, modifiers, "keyup");
        });
        
        qDebug() << "MacOSMenu: Forwarded key event to WebView using sendKeyboardEvent";
    }
}

// 文件菜单槽函数
void MacOSMenu::onNewFile()
{
    qDebug() << "MacOSMenu: New file requested";
    // 可以发送消息给WebView或处理新建文件逻辑
}

void MacOSMenu::onOpenFile()
{
    qDebug() << "MacOSMenu: Open file requested";
    // 可以发送消息给WebView或处理打开文件逻辑
}

void MacOSMenu::onSaveFile()
{
    qDebug() << "MacOSMenu: Save file requested";
    // 可以发送消息给WebView或处理保存文件逻辑
}

void MacOSMenu::onSaveAsFile()
{
    qDebug() << "MacOSMenu: Save as file requested";
    // 可以发送消息给WebView或处理另存为逻辑
}

void MacOSMenu::onQuit()
{
    qDebug() << "MacOSMenu: Quit requested";
    QApplication::quit();
}

// 编辑菜单槽函数 - 关键的快捷键处理
void MacOSMenu::onUndo()
{
    qDebug() << "MacOSMenu: Undo (Cmd+Z) triggered";
    forwardKeyEventToWebView(Qt::Key_Z, Qt::MetaModifier);
}

void MacOSMenu::onRedo()
{
    qDebug() << "MacOSMenu: Redo (Cmd+Shift+Z) triggered";
    forwardKeyEventToWebView(Qt::Key_Z, Qt::MetaModifier | Qt::ShiftModifier);
}

void MacOSMenu::onCut()
{
    qDebug() << "MacOSMenu: Cut (Cmd+X) triggered";
    forwardKeyEventToWebView(Qt::Key_X, Qt::MetaModifier);
}

void MacOSMenu::onCopy()
{
    qDebug() << "MacOSMenu: Copy (Cmd+C) triggered";
    forwardKeyEventToWebView(Qt::Key_C, Qt::MetaModifier);
}

void MacOSMenu::onPaste()
{
    qDebug() << "MacOSMenu: Paste (Cmd+V) triggered";
    forwardKeyEventToWebView(Qt::Key_V, Qt::MetaModifier);
}

void MacOSMenu::onSelectAll()
{
    qDebug() << "MacOSMenu: Select All (Cmd+A) triggered";
    forwardKeyEventToWebView(Qt::Key_A, Qt::MetaModifier);
}

// 视图菜单槽函数
void MacOSMenu::onToggleFullScreen()
{
    qDebug() << "MacOSMenu: Toggle full screen requested";
    // 处理全屏切换逻辑
}

void MacOSMenu::onZoomIn()
{
    qDebug() << "MacOSMenu: Zoom in requested";
    // 可以发送消息给WebView处理缩放
}

void MacOSMenu::onZoomOut()
{
    qDebug() << "MacOSMenu: Zoom out requested";
    // 可以发送消息给WebView处理缩放
}

void MacOSMenu::onActualSize()
{
    qDebug() << "MacOSMenu: Actual size requested";
    // 可以发送消息给WebView重置缩放
}

void MacOSMenu::onReload()
{
    qDebug() << "MacOSMenu: Reload (Cmd+R) triggered";
    if (m_webViewBridge) {
        // 发送重新加载消息给WebView
        MacOSWebViewBridge* macOSBridge = static_cast<MacOSWebViewBridge*>(m_webViewBridge);
        if (macOSBridge) {
            macOSBridge->reloadWebView();
        }
    }
}

// 窗口菜单槽函数
void MacOSMenu::onMinimize()
{
    qDebug() << "MacOSMenu: Minimize requested";
    // 处理窗口最小化逻辑
}

void MacOSMenu::onZoom()
{
    qDebug() << "MacOSMenu: Zoom requested";
    // 处理窗口缩放逻辑
}

void MacOSMenu::onBringAllToFront()
{
    qDebug() << "MacOSMenu: Bring all to front requested";
    // 处理窗口前置逻辑
}

// 帮助菜单槽函数
void MacOSMenu::onAbout()
{
    qDebug() << "MacOSMenu: About requested";
    // 显示关于对话框
}

void MacOSMenu::onHelp()
{
    qDebug() << "MacOSMenu: Help requested";
    // 显示帮助信息
}

#endif // Q_OS_MACOS