#ifndef MACOSMENU_H
#define MACOSMENU_H

#include <QObject>
#include <QMenuBar>
#include <QMenu>
#include <QAction>

#ifdef Q_OS_MACOS

class MacOSWebViewBridge;

/**
 * @brief macOS平台的原生菜单管理类
 * 处理标准的macOS菜单和快捷键，并将相关事件转发给WebView
 */
class MacOSMenu : public QObject
{
    Q_OBJECT

public:
    explicit MacOSMenu(QObject *parent = nullptr);
    ~MacOSMenu();

    /**
     * @brief 创建原生菜单栏
     */
    void createNativeMenuBar();

    /**
     * @brief 设置WebView桥接对象，用于转发快捷键事件
     * @param bridge WebView桥接对象
     */
    void setWebViewBridge(MacOSWebViewBridge* bridge);

private slots:
    // 文件菜单
    void onNewFile();
    void onOpenFile();
    void onSaveFile();
    void onSaveAsFile();
    void onQuit();

    // 编辑菜单 - 这些是关键的快捷键处理函数
    void onUndo();        // Cmd+Z
    void onRedo();        // Cmd+Shift+Z
    void onCut();         // Cmd+X
    void onCopy();        // Cmd+C
    void onPaste();       // Cmd+V
    void onSelectAll();   // Cmd+A

    // 视图菜单
    void onToggleFullScreen();
    void onZoomIn();
    void onZoomOut();
    void onActualSize();
    void onReload();       // 新增重新加载功能

    // 窗口菜单
    void onMinimize();
    void onZoom();
    void onBringAllToFront();

    // 帮助菜单
    void onAbout();
    void onHelp();

private:
    void createFileMenu();
    void createEditMenu();
    void createViewMenu();
    void createWindowMenu();
    void createHelpMenu();

    /**
     * @brief 将键盘事件转发给WebView
     * @param key Qt键码
     * @param modifiers 修饰键
     */
    void forwardKeyEventToWebView(int key, Qt::KeyboardModifiers modifiers);

private:
    QMenuBar* m_menuBar;
    MacOSWebViewBridge* m_webViewBridge;

    // 文件菜单
    QMenu* m_fileMenu;
    QAction* m_newAction;
    QAction* m_openAction;
    QAction* m_saveAction;
    QAction* m_saveAsAction;
    QAction* m_quitAction;

    // 编辑菜单
    QMenu* m_editMenu;
    QAction* m_undoAction;
    QAction* m_redoAction;
    QAction* m_cutAction;
    QAction* m_copyAction;
    QAction* m_pasteAction;
    QAction* m_selectAllAction;

    // 视图菜单
    QMenu* m_viewMenu;
    QAction* m_fullScreenAction;
    QAction* m_zoomInAction;
    QAction* m_zoomOutAction;
    QAction* m_actualSizeAction;
    QAction* m_reloadAction;    // 新增重新加载动作

    // 窗口菜单
    QMenu* m_windowMenu;
    QAction* m_minimizeAction;
    QAction* m_zoomAction;
    QAction* m_bringAllToFrontAction;

    // 帮助菜单
    QMenu* m_helpMenu;
    QAction* m_helpAction;
    QAction* m_aboutAction;
};

#endif // Q_OS_MACOS

#endif // MACOSMENU_H