#include "IOSWebViewBridge.h"

#ifdef Q_OS_IOS

#import <WebKit/WebKit.h>
#import <UIKit/UIKit.h>
#include <QDebug>
#include <QJsonDocument>
#include <QJsonObject>
#import "IOSViewController.h"

// Objective-C 委托类：处理 WKWebView 的消息与导航事件
// 说明：
// - JS→Native：在 didReceiveScriptMessage 中将消息转换为 QString，转交给 Qt 层处理
// - 页面加载完成：在 didFinishNavigation 中触发脚本注入与 webViewReady 信号（内部模式）
@interface IOSWebViewDelegate : NSObject <WKScriptMessageHandler, WKNavigationDelegate>
@property (nonatomic, assign) IOSWebViewBridge* bridge;
@end

@implementation IOSWebViewDelegate

- (void)userContentController:(WKUserContentController *)userContentController 
      didReceiveScriptMessage:(WKScriptMessage *)message {
    if (self.bridge && [message.name isEqualToString:@"bridge"]) {
        NSString* messageBody = nil;
        
        @try {
            if ([message.body isKindOfClass:[NSString class]]) {
                messageBody = (NSString*)message.body;
            } else if ([message.body isKindOfClass:[NSDictionary class]]) {
                NSError* error = nil;
                NSData* jsonData = [NSJSONSerialization dataWithJSONObject:message.body 
                                                                  options:0 
                                                                    error:&error];
                if (!error && jsonData) {
                    messageBody = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
                }
            }
            
            if (messageBody) {
                QString qMessage = QString::fromNSString(messageBody);
                // 使用Qt的信号槽机制安全地处理消息
                QMetaObject::invokeMethod(self.bridge, "handleNativeMessage", 
                                        Qt::QueuedConnection, 
                                        Q_ARG(QString, qMessage));
            }
        } @catch (NSException *exception) {
            NSLog(@"Exception in didReceiveScriptMessage: %@", exception.reason);
        }
    }
}

- (void)webView:(WKWebView *)webView didFinishNavigation:(WKNavigation *)navigation {
    if (self.bridge) {
        // 页面加载完成（内部模式）：重新注入桥接脚本并发射准备就绪信号
        self.bridge->injectBridgeScript();
        QMetaObject::invokeMethod(self.bridge, "webViewReady", Qt::QueuedConnection);
    }
}

@end

// 静态成员变量初始化
IOSWebViewBridge* IOSWebViewBridge::s_globalBridge = nullptr;

IOSWebViewBridge::IOSWebViewBridge(QObject* parent)
    : WebViewBridge(parent)
    , m_webView(nullptr)
    , m_delegate(nullptr)
    , m_viewController(nullptr)
    , m_containerWidget(nullptr)
    , m_isInitialized(false)
    , m_ownsWebView(true)
{
    // 内部模式：创建并配置 WKWebView 与委托
    setupWebView();
    // 设置为全局bridge实例
    setGlobalBridge(this);
}

IOSWebViewBridge::~IOSWebViewBridge()
{
    // 如果这是全局bridge，清除引用
    if (s_globalBridge == this) {
        s_globalBridge = nullptr;
    }
    
    // 仅在内部模式下释放 WebView，外部绑定时不持有生命周期
    if (m_webView && m_ownsWebView) {
        [m_webView release];
        m_webView = nullptr;
    }
    
    if (m_delegate) {
        [m_delegate release];
        m_delegate = nullptr;
    }
    
    if (m_viewController) {
        [m_viewController release];
        m_viewController = nullptr;
    }
}

void IOSWebViewBridge::setupWebView()
{
    // 内部模式下的 WKWebView 创建与基础配置（JS、偏好、进程池等）
    // 创建WKWebView配置
    WKWebViewConfiguration* config = [[WKWebViewConfiguration alloc] init];
    
    // 创建用户内容控制器
    WKUserContentController* userContentController = [[WKUserContentController alloc] init];
    
    // 创建委托对象
    m_delegate = [[IOSWebViewDelegate alloc] init];
    m_delegate.bridge = this;
    
    // 添加消息处理器（内部委托）：仅在内部模式使用
    [userContentController addScriptMessageHandler:m_delegate name:@"bridge"];
    
    config.userContentController = userContentController;
    
    // 允许内联播放媒体（iOS特有）
    config.allowsInlineMediaPlayback = YES;
    config.mediaTypesRequiringUserActionForPlayback = WKAudiovisualMediaTypeNone;
    
    // 设置JavaScript相关配置以避免JSC崩溃
    WKPreferences* preferences = [[WKPreferences alloc] init];
    preferences.javaScriptEnabled = YES;
    preferences.javaScriptCanOpenWindowsAutomatically = NO;
    config.preferences = preferences;
    
    // 设置进程池以提高稳定性
    config.processPool = [[WKProcessPool alloc] init];
    
    // 创建WKWebView
    CGRect frame = CGRectMake(0, 0, 375, 667); // iPhone默认尺寸
    m_webView = [[WKWebView alloc] initWithFrame:frame configuration:config];
    m_webView.navigationDelegate = m_delegate;
    
    // iOS特有设置
    m_webView.scrollView.bounces = NO;
    m_webView.scrollView.showsHorizontalScrollIndicator = NO;
    m_webView.scrollView.showsVerticalScrollIndicator = NO;
    
    // 将 WKWebView 嵌入到 Qt Widget 中（iOS 不直接使用 QWidget，此处保留为兼容）
    if (m_containerWidget) {
        // iOS平台不使用QWidget的winId，直接使用原生视图管理
        // 在iOS上，WebView将通过QML或其他原生方式管理
        qDebug() << "IOSWebViewBridge: WebView setup for iOS native view";
        
        // TODO: 实现iOS原生视图集成
        // 这里应该通过QML或其他iOS原生方式来管理WebView
    }
    
    [config release];
    [userContentController release];
}

void IOSWebViewBridge::initializeWebView()
{
    if (m_isInitialized) {
        return;
    }
    
    injectBridgeScript();
    m_isInitialized = true;
}

void IOSWebViewBridge::injectBridgeScript()
{
    // 注入桥接JavaScript代码（与macOS相同）
    QString bridgeScript = R"(
        window.NativeBridge = {
            sendMessage: function(method, params) {
                var message = {
                    method: method,
                    params: params || {}
                };
                window.webkit.messageHandlers.bridge.postMessage(JSON.stringify(message));
            },
            
            // 用于接收来自Native的消息
            onMessage: function(callback) {
                window.nativeMessageCallback = callback;
            }
        };
        
        // iOS特有的触摸事件优化
        document.addEventListener('touchstart', function() {}, {passive: true});
        document.addEventListener('touchmove', function() {}, {passive: true});
        
        // 通知Native桥接已准备就绪
        window.NativeBridge.sendMessage('bridgeReady', {});
    )";
    
    evaluateJavaScript(bridgeScript);
}

void IOSWebViewBridge::sendMessage(const QString& method, const QJsonObject& params)
{
    // 将 Native 响应封装为 JSON，交由页面回调处理
    QJsonObject message;
    message["method"] = method;
    message["params"] = params;
    
    QJsonDocument doc(message);
    QString jsonString = doc.toJson(QJsonDocument::Compact);
    
    // 调用JavaScript回调函数
    QString script = QString("if (window.nativeMessageCallback) { window.nativeMessageCallback(%1); }")
                        .arg(jsonString);
    
    evaluateJavaScript(script);
}

void IOSWebViewBridge::evaluateJavaScript(const QString& script)
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    // 添加JavaScript执行的安全检查
    if (script.isEmpty()) {
        qWarning() << "Empty JavaScript script";
        return;
    }
    
    NSString* nsScript = script.toNSString();
    
    // 使用主线程执行 JavaScript 以避免 JSC 崩溃
    dispatch_async(dispatch_get_main_queue(), ^{
        [m_webView evaluateJavaScript:nsScript completionHandler:^(id result, NSError* error) {
            if (error) {
                qWarning() << "JavaScript execution error:" << QString::fromNSString(error.localizedDescription);
                qWarning() << "Error code:" << error.code;
                qWarning() << "Script:" << script;
            } else {
                qDebug() << "JavaScript executed successfully";
            }
        }];
    });
}

void IOSWebViewBridge::loadHtml(const QString& html)
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    NSString* nsHtml = html.toNSString();
    [m_webView loadHTMLString:nsHtml baseURL:nil];
}

void IOSWebViewBridge::loadUrl(const QString& url)
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    NSURL* nsUrl = [NSURL URLWithString:url.toNSString()];
    NSURLRequest* request = [NSURLRequest requestWithURL:nsUrl];
    [m_webView loadRequest:request];
}

QObject* IOSWebViewBridge::getWebViewWidget()
{
    return m_containerWidget;
}

void IOSWebViewBridge::handleNativeMessage(const QString& message)
{
    // 调用基类的消息处理方法
    handleMessage(message);
}

void IOSWebViewBridge::setGlobalBridge(IOSWebViewBridge* bridge)
{
    s_globalBridge = bridge;
}

IOSWebViewBridge* IOSWebViewBridge::getGlobalBridge()
{
    return s_globalBridge;
}

void IOSWebViewBridge::handleAppDelegateMessage(const QString& message)
{
    IOSWebViewBridge* bridge = getGlobalBridge();
    if (bridge) {
        // 通过全局 bridge 转发消息到 Qt C++ 层
        bridge->handleNativeMessage(message);
    } else {
        qWarning() << "IOSWebViewBridge: No global bridge instance available, message not forwarded:" << message;
    }
}

void IOSWebViewBridge::showWebView()
{
    if (!m_webView) {
        qWarning() << "WebView not initialized";
        return;
    }
    
    // 创建专用的视图控制器
    if (!m_viewController) {
        m_viewController = [[IOSViewController alloc] init];
    }
    
    // 设置WebView到视图控制器
    [m_viewController setupWebView:m_webView];
    
    // 使用dispatch_async确保在主线程上设置根视图控制器
    dispatch_async(dispatch_get_main_queue(), ^{
        UIViewController* rootViewController = nil;
        
        if (@available(iOS 13.0, *)) {
            // iOS 13+ 使用 connectedScenes
            for (UIWindowScene* windowScene in [UIApplication sharedApplication].connectedScenes) {
                if (windowScene.activationState == UISceneActivationStateForegroundActive) {
                    for (UIWindow* window in windowScene.windows) {
                        if (window.isKeyWindow) {
                            window.rootViewController = m_viewController;
                            [window makeKeyAndVisible];
                            qDebug() << "Set IOSViewController as root view controller (iOS 13+)";
                            return;
                        }
                    }
                }
            }
        } else {
            // iOS 12 及以下版本使用 keyWindow
            UIWindow* keyWindow = [UIApplication sharedApplication].keyWindow;
            if (keyWindow) {
                keyWindow.rootViewController = m_viewController;
                [keyWindow makeKeyAndVisible];
                qDebug() << "Set IOSViewController as root view controller (iOS 12)";
                return;
            }
        }
        
        qWarning() << "Failed to find key window to set root view controller";
    });
}

void IOSWebViewBridge::useExternalWebView(WKWebView* webView)
{
    if (!webView) {
        qWarning() << "useExternalWebView called with null WKWebView";
        return;
    }
    
    // 如果已创建内部WebView，释放之并清理委托，避免重复持有与消息处理
    if (m_webView && m_ownsWebView) {
        [m_webView release];
        m_webView = nullptr;
    }
    
    if (m_delegate) {
        // 我们不在外部WebView上注册此委托，释放内部创建的委托即可
        [m_delegate release];
        m_delegate = nullptr;
    }
    
    m_webView = webView;
    m_ownsWebView = false;
    
    qDebug() << "IOSWebViewBridge: Bound to external WKWebView instance";
}

#endif // Q_OS_IOS