#ifndef IOSWEBVIEWBRIDGE_H
#define IOSWEBVIEWBRIDGE_H

#include "../../bridge/WebViewBridge.h"

#ifdef Q_OS_IOS

#include <QObject>

// 前向声明Objective-C类
#ifdef __OBJC__
@class WKWebView;
@class IOSWebViewDelegate;
@class UIView;
@class IOSViewController;
#else
class WKWebView;
class IOSWebViewDelegate;
class UIView;
class IOSViewController;
#endif

/**
 * @brief iOS 平台的 WKWebView 桥接实现
 *
 * 职责说明：
 * - 统一管理 Native↔Web 的双向通信（Qt C++ ↔ WKWebView JavaScript）
 * - 封装消息收发、脚本注入、视图展示等操作
 * - 兼容两种 WebView 来源：内部创建（默认）或外部绑定（推荐：AppDelegate 创建并展示的）
 *
 * 设计要点：
 * - JS→Native：通过 `window.webkit.messageHandlers.bridge.postMessage(...)` 由 App 层转发至 Qt 层
 * - Native→JS：通过 `evaluateJavaScript(...)` 调用 `window.nativeMessageCallback(...)`
 * - 脚本注入：`injectBridgeScript()` 负责设置 `window.NativeBridge` 与回调绑定
 * - 全局实例：通过 `setGlobalBridge/getGlobalBridge` 提供系统级访问，便于 AppDelegate 转发消息
 * - 所有权控制：若使用 `useExternalWebView` 绑定外部 WebView，则桥接不持有其生命周期，避免双重释放
 */
class IOSWebViewBridge : public WebViewBridge
{
    Q_OBJECT

public:
    explicit IOSWebViewBridge(QObject* parent = nullptr);
    ~IOSWebViewBridge() override;

    // 实现基类的纯虚函数
    /**
     * @brief 发送消息到 Web（Native → JS）
     *
     * 通过调用 `window.nativeMessageCallback(message)` 将响应或通知传回页面。
     * 要求：页面在注入脚本后已注册 `onMessage(cb)` 并持有回调。
     */
    void sendMessage(const QString& method, const QJsonObject& params = QJsonObject()) override;
    /**
     * @brief 执行任意 JavaScript 脚本
     *
     * 在主线程安全执行，并输出错误日志；空脚本将被忽略。
     */
    void evaluateJavaScript(const QString& script) override;
    /**
     * @brief 初始化 WebView（内部模式）
     *
     * 仅在桥接内部创建 WebView 时使用；若绑定外部 WebView，通常无需调用。
     */
    void initializeWebView() override;
    /**
     * @brief 加载 HTML 字符串（内部或外部绑定的 WebView 都可用）
     */
    void loadHtml(const QString& html) override;
    /**
     * @brief 加载 URL（内部或外部绑定的 WebView 都可用）
     */
    void loadUrl(const QString& url) override;

    /**
     * @brief 获取WebView的原生视图
     * @return QObject指针，iOS平台不使用QWidget
     */
    /**
     * @brief 获取 WebView 的容器（iOS 不使用 QWidget，此处返回容器对象占位）
     */
    QObject* getWebViewWidget();

    /**
     * @brief 处理从WKWebView接收到的消息（由Objective-C代码调用）
     * @param message 消息内容
     */
    /**
     * @brief 处理从 WKWebView 接收到的消息（由 ObjC 委托或 AppDelegate 转发）
     * @param message JSON 格式的消息字符串
     */
    void handleNativeMessage(const QString& message);

    /**
     * @brief 注入桥接脚本（供delegate调用）
     */
    /**
     * @brief 注入桥接脚本：定义 `window.NativeBridge` 与 `onMessage` 回调绑定
     *
     * 幂等：重复注入将覆盖旧定义，以确保页面保持最新桥接状态。
     */
    void injectBridgeScript();

    /**
     * @brief 显示WebView
     */
    /**
     * @brief 展示 WebView（内部模式）：通过专用 `IOSViewController` 承载展示
     */
    void showWebView();

    /**
     * @brief 绑定外部创建的WKWebView（例如由AppDelegate创建并显示的）
     * @param webView 外部WKWebView实例指针
     *
     * 绑定后，本桥接不再拥有该WebView的生命周期，也不在其上注册自身的委托。
     * 发送消息与执行JavaScript将作用于该外部WebView。
     */
    void useExternalWebView(WKWebView* webView);

    /**
     * @brief 设置全局bridge实例（供AppDelegate使用）
     * @param bridge bridge实例指针
     */
    /**
     * @brief 设置全局 bridge 实例（供 AppDelegate 使用）
     */
    static void setGlobalBridge(IOSWebViewBridge* bridge);

    /**
     * @brief 获取全局bridge实例
     * @return bridge实例指针
     */
    /**
     * @brief 获取全局 bridge 实例
     */
    static IOSWebViewBridge* getGlobalBridge();

    /**
     * @brief 处理来自AppDelegate的消息（全局函数，供Objective-C调用）
     * @param message JSON格式的消息字符串
     */
    /**
     * @brief 处理来自 AppDelegate 的消息：使用全局实例进行转发
     */
    static void handleAppDelegateMessage(const QString& message);

private:
    void setupWebView();

private:
    WKWebView* m_webView;
    IOSWebViewDelegate* m_delegate;
    IOSViewController* m_viewController;
    QObject* m_containerWidget;
    bool m_isInitialized;
    bool m_ownsWebView;

    // 全局bridge实例（供AppDelegate使用）
    static IOSWebViewBridge* s_globalBridge;
};

#endif // Q_OS_IOS

#endif // IOSWEBVIEWBRIDGE_H