#ifdef Q_OS_IOS

#import "AppDelegate.h"
#import "IOSViewController.h"
#import <WebKit/WebKit.h>
// 导入C++桥接头文件以转发消息到Qt层
#include "IOSWebViewBridge.h"
#include <QCoreApplication>
#include "demo/DemoModel.h"
#include "demo/DemoController.h"
#include "platform/ios/IOSView.h"
#include <QString>

/**
 * iOS 应用入口与 WebView 创建封装说明
 *
 * 职责：
 * - 作为唯一的 WKWebView 所有者，负责创建、展示与生命周期管理
 * - 统一 JS↔Native 双向通信通道，避免出现“双 WebView/双桥接/重复脚本注入”问题
 *
 * 启动与通信流程：
 * 1) didFinishLaunchingWithOptions：
 *    - 创建 UIWindow 与 IOSViewController
 *    - 配置 WKWebView（添加 ScriptMessageHandler:"bridge"，注入 window.NativeBridge 脚本）
 *    - 创建与展示 WKWebView
 *    - 初始化 Qt 逻辑层（QCoreApplication + DemoMVC）
 *    - 将全局 IOSWebViewBridge 绑定到当前显示的外部 WKWebView（useExternalWebView）
 *    - 加载 HTML 页面
 * 2) didFinishNavigation：
 *    - 注入桥接脚本（防止某些页面跳转后丢失）
 *    - 通过 QMetaObject::invokeMethod 触发 webViewReady，通知 Qt 层页面可用
 *
 * 双向通信：
 * - JS→Native：window.NativeBridge.sendMessage(...) → WKScriptMessageHandler("bridge")
 *              → AppDelegate.didReceiveScriptMessage → IOSWebViewBridge::handleAppDelegateMessage
 * - Native→JS：IOSWebViewBridge::sendMessage/evaluateJavaScript
 *              → window.nativeMessageCallback({ method, params })
 *
 * 远程调试：
 * - iOS 16.4+ 可开启 webView.inspectable = YES；配合 Safari “开发”菜单进行调试
 */

@interface AppDelegate ()
@property (strong, nonatomic) IOSViewController *viewController;
@property (strong, nonatomic) WKWebView *webView;
@end

@implementation AppDelegate

- (BOOL)application:(UIApplication *)application didFinishLaunchingWithOptions:(NSDictionary *)launchOptions {
    NSLog(@"AppDelegate: didFinishLaunchingWithOptions called");
    // 启动流程步骤概览：创建窗口 → 配置与注入 → 创建/展示 WebView → 初始化 Qt → 绑定桥接 → 加载页面
    
    // 创建窗口
    self.window = [[UIWindow alloc] initWithFrame:[[UIScreen mainScreen] bounds]];
    self.window.backgroundColor = [UIColor redColor]; // 临时设为红色以便调试
    
    NSLog(@"AppDelegate: Window created with frame: %@", NSStringFromCGRect(self.window.frame));
    
    // 创建WebView配置
    WKWebViewConfiguration *config = [[WKWebViewConfiguration alloc] init];
    
    // 添加JavaScript消息处理器
    WKUserContentController *userContentController = [[WKUserContentController alloc] init];
    [userContentController addScriptMessageHandler:self name:@"bridge"];
    // 注入 NativeBridge 脚本，提供 window.NativeBridge
    NSString *bridgeScript = @"(function(){ window.NativeBridge = { sendMessage: function(method, params){ var msg = { method: method, params: params || {} }; window.webkit.messageHandlers.bridge.postMessage(JSON.stringify(msg)); }, onMessage: function(cb){ window.nativeMessageCallback = cb; } }; window.NativeBridge.sendMessage('bridgeReady', {}); })();";
    WKUserScript *bridgeUserScript = [[WKUserScript alloc] initWithSource:bridgeScript injectionTime:WKUserScriptInjectionTimeAtDocumentStart forMainFrameOnly:YES];
    [userContentController addUserScript:bridgeUserScript];
    config.userContentController = userContentController;
    
    // 创建WebView
    self.webView = [[WKWebView alloc] initWithFrame:self.window.bounds configuration:config];
    self.webView.navigationDelegate = self;
    self.webView.backgroundColor = [UIColor greenColor]; // 临时设为绿色以便调试
    
    // 在支持的系统上开启远程调试检查开关
    if (@available(iOS 16.4, *)) {
        if ([self.webView respondsToSelector:@selector(setInspectable:)]) {
            [self.webView setInspectable:YES];
            NSLog(@"WKWebView: inspectable enabled for remote debugging");
        }
    }
    
    NSLog(@"AppDelegate: WebView created with frame: %@", NSStringFromCGRect(self.webView.frame));
    
    // 创建视图控制器
    self.viewController = [[IOSViewController alloc] init];
    NSLog(@"AppDelegate: ViewController created");
    
    [self.viewController setupWebView:self.webView];
    NSLog(@"AppDelegate: WebView setup completed");
    
    // 设置根视图控制器
    self.window.rootViewController = self.viewController;
    NSLog(@"AppDelegate: Root view controller set");
    
    [self.window makeKeyAndVisible];
    NSLog(@"AppDelegate: Window made key and visible");

    // 初始化 Qt 逻辑层（不影响 WebView 启动流程）
    @try {
        NSLog(@"AppDelegate: Initializing Qt logic layer");
        // 确保存在一个 Qt Core 应用实例
        if (!QCoreApplication::instance()) {
            int argc = 0;
            char* argv[] = { nullptr };
            new QCoreApplication(argc, argv);
        }

        // 创建 MVC 组件并绑定桥接（保证全局桥接实例存在）
        DemoModel* model = new DemoModel();
        DemoController* controller = new DemoController();
        IOSView* iosView = new IOSView();
        controller->setModel(model);
        controller->setWebViewBridge(iosView->getWebViewBridge());

        // 将桥接绑定到当前显示的外部WKWebView，统一双向通信通道
        IOSWebViewBridge* globalBridge = IOSWebViewBridge::getGlobalBridge();
        if (globalBridge) {
            globalBridge->useExternalWebView(self.webView);
            NSLog(@"AppDelegate: Global bridge bound to external WKWebView");
        } else {
            NSLog(@"AppDelegate: Warning - Global bridge not available to bind");
        }

        controller->initialize();
        NSLog(@"AppDelegate: Qt logic layer initialized successfully");
    } @catch (NSException *exception) {
        NSLog(@"AppDelegate: Exception during Qt logic init: %@", exception.reason);
    }

    // 延迟加载HTML内容，确保视图已经显示
    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(0.5 * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
        [self loadWebContent];
    });
    
    return YES;
}

- (void)loadWebContent {
    NSLog(@"AppDelegate: loadWebContent called");
    
    // 加载HTML文件
    NSString *htmlPath = [[NSBundle mainBundle] pathForResource:@"index" ofType:@"html" inDirectory:@"web"];
    NSLog(@"AppDelegate: HTML path: %@", htmlPath);
    
    if (htmlPath) {
        NSError *error = nil;
        NSString *htmlContent = [NSString stringWithContentsOfFile:htmlPath encoding:NSUTF8StringEncoding error:&error];
        if (htmlContent) {
            NSLog(@"AppDelegate: Loading HTML content from file");
            [self.webView loadHTMLString:htmlContent baseURL:nil];
            return;
        } else {
            NSLog(@"AppDelegate: Failed to read HTML file: %@", error.localizedDescription);
        }
    }
    
    // 如果文件加载失败，使用默认内容
    NSString *defaultHTML = @"<html><head><meta name='viewport' content='width=device-width, initial-scale=1.0'></head><body style='margin:0; padding:20px; font-family:Arial; background-color:#f0f0f0;'><h1 style='color:#333;'>GGTPassword</h1><p style='color:#666;'>WebView Ready and Working!</p><button onclick='alert(\"Button clicked!\")'>Test Button</button></body></html>";
    NSLog(@"AppDelegate: Loading default HTML content");
    [self.webView loadHTMLString:defaultHTML baseURL:nil];
}

#pragma mark - WKScriptMessageHandler

- (void)userContentController:(WKUserContentController *)userContentController didReceiveScriptMessage:(WKScriptMessage *)message {
    if ([message.name isEqualToString:@"bridge"]) {
        NSLog(@"AppDelegate: Received message from WebView: %@", message.body);
        // JS→Native 流程：将消息体转换为 NSString → QString → 交由 Qt 层处理
        
        NSString* messageBody = nil;
        
        @try {
            // 将消息体转换为NSString
            if ([message.body isKindOfClass:[NSString class]]) {
                messageBody = (NSString*)message.body;
            } else if ([message.body isKindOfClass:[NSDictionary class]]) {
                NSError* error = nil;
                NSData* jsonData = [NSJSONSerialization dataWithJSONObject:message.body 
                                                                  options:0 
                                                                    error:&error];
                if (!error && jsonData) {
                    messageBody = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
                } else {
                    NSLog(@"AppDelegate: Failed to serialize message body: %@", error.localizedDescription);
                }
            }
            
            // 转发消息到Qt C++层（通过IOSWebViewBridge）
            if (messageBody) {
                QString qMessage = QString::fromNSString(messageBody);
                IOSWebViewBridge::handleAppDelegateMessage(qMessage);
                NSLog(@"AppDelegate: Message forwarded to Qt C++ layer");
            } else {
                NSLog(@"AppDelegate: Warning - messageBody is nil, message not forwarded");
            }
        } @catch (NSException *exception) {
            NSLog(@"AppDelegate: Exception in didReceiveScriptMessage: %@", exception.reason);
        }
    }
}

#pragma mark - WKNavigationDelegate

- (void)webView:(WKWebView *)webView didStartProvisionalNavigation:(WKNavigation *)navigation {
    NSLog(@"WebView: didStartProvisionalNavigation");
}

- (void)webView:(WKWebView *)webView didCommitNavigation:(WKNavigation *)navigation {
    NSLog(@"WebView: didCommitNavigation");
}

- (void)webView:(WKWebView *)webView didFinishNavigation:(WKNavigation *)navigation {
    NSLog(@"WebView: didFinishNavigation - content loaded successfully");

    // 与macOS保持一致：页面完成后注入桥接脚本并触发webViewReady
    IOSWebViewBridge* bridge = IOSWebViewBridge::getGlobalBridge();
    if (bridge) {
        bridge->injectBridgeScript();
        QMetaObject::invokeMethod(bridge, "webViewReady", Qt::QueuedConnection);
        NSLog(@"AppDelegate: Bridge script injected and webViewReady emitted");
    } else {
        NSLog(@"AppDelegate: Warning - Global bridge missing in didFinishNavigation");
    }
}

- (void)webView:(WKWebView *)webView didFailNavigation:(WKNavigation *)navigation withError:(NSError *)error {
    NSLog(@"WebView: didFailNavigation with error: %@", error.localizedDescription);
}

- (void)webView:(WKWebView *)webView didFailProvisionalNavigation:(WKNavigation *)navigation withError:(NSError *)error {
    NSLog(@"WebView: didFailProvisionalNavigation with error: %@", error.localizedDescription);
}

@end

#endif // Q_OS_IOS