#ifndef ANDROIDWEBVIEWBRIDGE_H
#define ANDROIDWEBVIEWBRIDGE_H

#include "../../bridge/WebViewBridge.h"

#ifdef Q_OS_ANDROID

#include <QObject>
#include <QJniObject>
#include <QJniEnvironment>
#include <QtCore/qjniobject.h>

/**
 * @brief Android平台的WebView桥接实现
 * 使用Android WebView和JavaScriptInterface进行通信
 */
class AndroidWebViewBridge : public WebViewBridge
{
    Q_OBJECT

public:
    explicit AndroidWebViewBridge(QObject* parent = nullptr);
    ~AndroidWebViewBridge() override;

    // 实现基类的纯虚函数
    void sendMessage(const QString& method, const QJsonObject& params = QJsonObject()) override;
    void evaluateJavaScript(const QString& script) override;
    void initializeWebView() override;
    void loadHtml(const QString& html) override;
    void loadUrl(const QString& url) override;

    /**
     * @brief 获取WebView对象
     * @return QJniObject WebView实例
     */
    QJniObject getWebViewObject() const;

    /**
     * @brief 处理从Android WebView接收到的消息（由JNI调用）
     * @param message 消息内容
     */
    void handleWebViewMessage(const QString& message);

    /**
     * @brief 设置WebView的父Activity
     * @param activity Android Activity对象
     */
    void setParentActivity(const QJniObject& activity);

    /**
     * @brief 调整WebView大小和位置
     * @param x X坐标
     * @param y Y坐标
     * @param width 宽度
     * @param height 高度
     */
    void resizeWebView(int x, int y, int width, int height);

private slots:
    void onApplicationStateChanged(Qt::ApplicationState state);

private:
    /**
     * @brief 初始化JNI环境和Java类
     */
    bool initializeJNI();

    /**
     * @brief 创建Android WebView实例
     */
    bool createWebView();

    /**
     * @brief 设置WebView配置
     */
    void setupWebViewSettings();

    /**
     * @brief 注入JavaScript桥接代码
     */
    void injectBridgeScript();

    /**
     * @brief 向WebView发送消息
     * @param method 方法名
     * @param params 参数
     */
    void sendMessageToWeb(const QString& method, const QJsonObject& params);

    /**
     * @brief 处理从WebView接收的消息
     * @param message JSON格式的消息
     */
    void handleWebMessage(const QString& message);

public:
    /**
     * @brief 清理资源
     */
    void cleanup();

private:
    QJniObject m_webView;           // Android WebView对象
    QJniObject m_activity;          // 父Activity对象
    QJniObject m_webViewClient;     // WebViewClient对象
    QJniObject m_webChromeClient;   // WebChromeClient对象
    QJniObject m_jsInterface;       // JavaScript接口对象
    
    bool m_isInitialized;                  // 是否已初始化
    QString m_pendingUrl;                  // 待加载的URL
    QString m_pendingHtml;                 // 待加载的HTML
    
    // WebView配置
    int m_webViewX;
    int m_webViewY;
    int m_webViewWidth;
    int m_webViewHeight;
};

// JNI回调函数声明
extern "C" {
    /**
     * @brief JNI回调：处理来自WebView的消息
     * @param env JNI环境
     * @param thiz Java对象
     * @param message 消息内容
     */
    JNIEXPORT void JNICALL
    Java_org_qtproject_qt_android_bindings_QtActivity_onWebViewMessage(
        JNIEnv *env, jobject thiz, jstring message);

    /**
     * @brief JNI回调：WebView页面加载完成
     * @param env JNI环境
     * @param thiz Java对象
     * @param url 加载的URL
     */
    JNIEXPORT void JNICALL
    Java_org_qtproject_qt_android_bindings_QtActivity_onWebViewPageFinished(
        JNIEnv *env, jobject thiz, jstring url);

    /**
     * @brief JNI回调：WebView页面开始加载
     * @param env JNI环境
     * @param thiz Java对象
     * @param url 加载的URL
     */
    JNIEXPORT void JNICALL
    Java_org_qtproject_qt_android_bindings_QtActivity_onWebViewPageStarted(
        JNIEnv *env, jobject thiz, jstring url);
}

#endif // Q_OS_ANDROID

#endif // ANDROIDWEBVIEWBRIDGE_H