#include "AndroidWebViewBridge.h"

#ifdef Q_OS_ANDROID

#include <QDebug>
#include <QJsonDocument>
#include <QJsonObject>
#include <QCoreApplication>
#include <QGuiApplication>
#include <QTimer>
#include <QtCore/qjniobject.h>
#include <QJniEnvironment>

static AndroidWebViewBridge* g_androidWebViewBridge = nullptr;

AndroidWebViewBridge::AndroidWebViewBridge(QObject* parent)
    : WebViewBridge(parent)
    , m_isInitialized(false)
    , m_webViewX(0)
    , m_webViewY(0)
    , m_webViewWidth(800)
    , m_webViewHeight(600)
{
    g_androidWebViewBridge = this;
    connect(qApp, &QGuiApplication::applicationStateChanged,
            this, &AndroidWebViewBridge::onApplicationStateChanged);
}

AndroidWebViewBridge::~AndroidWebViewBridge()
{
    cleanup();
    g_androidWebViewBridge = nullptr;
}

void AndroidWebViewBridge::sendMessage(const QString& method, const QJsonObject& params)
{
    sendMessageToWeb(method, params);
}

void AndroidWebViewBridge::evaluateJavaScript(const QString& script)
{
    if (!m_webView.isValid()) {
        qWarning() << "AndroidWebViewBridge: WebView is not valid, cannot evaluate JS";
        return;
    }

    QString preview = script.left(200);
    if (script.length() > 200) preview += "...";
    qDebug() << "AndroidWebViewBridge: evaluateJavaScript() called, length:" << script.length() << ", preview:" << preview;

    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this, script]() {
        QJniEnvironment env;
        jstring jScript = env->NewStringUTF(script.toUtf8().constData());
        // 使用自定义回调记录结果
        QJniObject callback("com/ggt/ggtpassword/JsResultCallback", "()V");
        if (callback.isValid()) {
            m_webView.callMethod<void>("evaluateJavascript", "(Ljava/lang/String;Landroid/webkit/ValueCallback;)V",
                                       jScript, callback.object<jobject>());
            qDebug() << "AndroidWebViewBridge: evaluateJavascript() dispatched with ValueCallback";
        } else {
            m_webView.callMethod<void>("evaluateJavascript", "(Ljava/lang/String;Landroid/webkit/ValueCallback;)V",
                                       jScript, nullptr);
            qWarning() << "AndroidWebViewBridge: JsResultCallback invalid, dispatched with null callback";
        }
        env->DeleteLocalRef(jScript);

        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: evaluateJavascript threw exception";
        }
    });
}

void AndroidWebViewBridge::initializeWebView()
{
    if (m_isInitialized) {
        qDebug() << "AndroidWebViewBridge: Already initialized";
        return;
    }

    qDebug() << "AndroidWebViewBridge: Initializing WebView...";

    if (!initializeJNI()) {
        qWarning() << "AndroidWebViewBridge: Failed to initialize JNI";
        return;
    }

    // 启用 WebView 内容调试，便于 chrome://inspect 调试
    QJniObject::callStaticMethod<void>("android/webkit/WebView", "setWebContentsDebuggingEnabled", "(Z)V", true);

    bool created = false;
    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([&]() {
        QJniEnvironment env;
        qDebug() << "AndroidWebViewBridge: [UI] Enter initializeWebView block";
        created = createWebView();
        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: [UI] Exception after createWebView";
        }
        if (!created) {
            qWarning() << "AndroidWebViewBridge: createWebView() returned false";
            qDebug() << "AndroidWebViewBridge: [UI] Exit initializeWebView block (create failed)";
            return;
        }
        qDebug() << "AndroidWebViewBridge: createWebView() returned true, applying settings";
        setupWebViewSettings();
        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: [UI] Exception after setupWebViewSettings";
        }
        qDebug() << "AndroidWebViewBridge: setupWebViewSettings() done, injecting bridge script";
        injectBridgeScript();
        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: [UI] Exception after injectBridgeScript";
        }
        qDebug() << "AndroidWebViewBridge: [UI] Exit initializeWebView block";
    }).waitForFinished();

    if (!created) {
        qWarning() << "AndroidWebViewBridge: Failed to create WebView";
        return;
    }

    m_isInitialized = true;

    // 如果有待加载的内容，现在加载（在UI线程执行）
    if (!m_pendingUrl.isEmpty()) {
        qDebug() << "AndroidWebViewBridge: Loading pending URL:" << m_pendingUrl;
        loadUrl(m_pendingUrl);
        m_pendingUrl.clear();
    } else if (!m_pendingHtml.isEmpty()) {
        qDebug() << "AndroidWebViewBridge: Loading pending HTML, length:" << m_pendingHtml.length();
        loadHtml(m_pendingHtml);
        m_pendingHtml.clear();
    }

    emit webViewReady();
    qDebug() << "AndroidWebViewBridge: WebView initialized successfully";
}

void AndroidWebViewBridge::loadHtml(const QString& html)
{
    qDebug() << "AndroidWebViewBridge: loadHtml() called, initialized:" << m_isInitialized << ", webView valid:" << m_webView.isValid() << ", length:" << html.length();
    if (!m_isInitialized || !m_webView.isValid()) {
        m_pendingHtml = html;
        qDebug() << "AndroidWebViewBridge: WebView not ready, storing HTML for later loading";
        return;
    }

    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this, html]() {
        QJniEnvironment env;
        QString htmlPreview = html.left(200);
        if (html.length() > 200) htmlPreview += "...";
        qDebug() << "AndroidWebViewBridge: loadHtml preview:" << htmlPreview;
        jstring jHtml = env->NewStringUTF(html.toUtf8().constData());
        jstring jMimeType = env->NewStringUTF("text/html");
        jstring jEncoding = env->NewStringUTF("UTF-8");
        jstring jBaseUrl = env->NewStringUTF("about:blank");

        // 使用 loadDataWithBaseURL，避免部分设备对 loadData 渲染异常
        qDebug() << "AndroidWebViewBridge: Calling loadDataWithBaseURL(about:blank)";
        m_webView.callMethod<void>("loadDataWithBaseURL", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V",
                                   jBaseUrl, jHtml, jMimeType, jEncoding, nullptr);

        env->DeleteLocalRef(jHtml);
        env->DeleteLocalRef(jMimeType);
        env->DeleteLocalRef(jEncoding);
        env->DeleteLocalRef(jBaseUrl);

        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: Failed to load HTML";
        } else {
            qDebug() << "AndroidWebViewBridge: HTML loaded via loadDataWithBaseURL";
        }
    });
}

void AndroidWebViewBridge::loadUrl(const QString& url)
{
    qDebug() << "AndroidWebViewBridge: loadUrl() called, initialized:" << m_isInitialized << ", webView valid:" << m_webView.isValid() << ", url:" << url;
    if (!m_isInitialized || !m_webView.isValid()) {
        m_pendingUrl = url;
        qDebug() << "AndroidWebViewBridge: WebView not ready, storing URL for later loading:" << url;
        return;
    }

    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this, url]() {
        QJniEnvironment env;
        qDebug() << "AndroidWebViewBridge: [UI] Enter loadUrl block ->" << url;
        jstring jUrl = env->NewStringUTF(url.toUtf8().constData());
        qDebug() << "AndroidWebViewBridge: Calling WebView.loadUrl(" << url << ")";
        m_webView.callMethod<void>("loadUrl", "(Ljava/lang/String;)V", jUrl);
        env->DeleteLocalRef(jUrl);

        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: Failed to load URL:" << url;
        } else {
            qDebug() << "AndroidWebViewBridge: loadUrl request sent";
        }
        qDebug() << "AndroidWebViewBridge: [UI] Exit loadUrl block";
    });
}

bool AndroidWebViewBridge::createWebView()
{
    QJniEnvironment env;

    qDebug() << "AndroidWebViewBridge: createWebView() entered, activity valid:" << m_activity.isValid();
    if (!m_activity.isValid()) {
        qWarning() << "AndroidWebViewBridge: Activity invalid, cannot create WebView";
        return false;
    }

    // 创建WebView实例
    m_webView = QJniObject("android/webkit/WebView", "(Landroid/content/Context;)V",
                                  m_activity.object<jobject>());

    if (env->ExceptionCheck()) {
        env->ExceptionDescribe();
        env->ExceptionClear();
        qWarning() << "AndroidWebViewBridge: Exception thrown during WebView construction";
    }

    if (!m_webView.isValid()) {
        qWarning() << "AndroidWebViewBridge: Failed to create WebView instance";
        return false;
    }

    qDebug() << "AndroidWebViewBridge: WebView instance created";

    // 使用自定义 LoggingWebViewClient，记录加载事件与错误
    QJniObject webViewClient("com/ggt/ggtpassword/LoggingWebViewClient", "()V");
    if (webViewClient.isValid()) {
        m_webViewClient = webViewClient;
        m_webView.callMethod<void>("setWebViewClient", "(Landroid/webkit/WebViewClient;)V",
                                   m_webViewClient.object<jobject>());
        qDebug() << "AndroidWebViewBridge: LoggingWebViewClient set";
    } else {
        qWarning() << "AndroidWebViewBridge: Failed to create LoggingWebViewClient, falling back to standard";
        QJniObject fallbackClient("android/webkit/WebViewClient", "()V");
        if (fallbackClient.isValid()) {
            m_webViewClient = fallbackClient;
            m_webView.callMethod<void>("setWebViewClient", "(Landroid/webkit/WebViewClient;)V",
                                       m_webViewClient.object<jobject>());
            qDebug() << "AndroidWebViewBridge: Standard WebViewClient set";
        }
    }

    // 使用自定义 LoggingWebChromeClient，记录 console 与进度
    QJniObject webChromeClient("com/ggt/ggtpassword/LoggingWebChromeClient", "()V");
    if (webChromeClient.isValid()) {
        m_webChromeClient = webChromeClient;
        m_webView.callMethod<void>("setWebChromeClient", "(Landroid/webkit/WebChromeClient;)V",
                                   m_webChromeClient.object<jobject>());
        qDebug() << "AndroidWebViewBridge: LoggingWebChromeClient set";
    } else {
        qWarning() << "AndroidWebViewBridge: Failed to create LoggingWebChromeClient, falling back to standard";
        QJniObject fallbackChrome("android/webkit/WebChromeClient", "()V");
        if (fallbackChrome.isValid()) {
            m_webChromeClient = fallbackChrome;
            m_webView.callMethod<void>("setWebChromeClient", "(Landroid/webkit/WebChromeClient;)V",
                                       m_webChromeClient.object<jobject>());
            qDebug() << "AndroidWebViewBridge: Standard WebChromeClient set";
        }
    }

    if (env->ExceptionCheck()) {
        env->ExceptionDescribe();
        env->ExceptionClear();
        qWarning() << "AndroidWebViewBridge: Exception occurred while setting WebView clients";
    }

    // 注入 JavascriptInterface 供 JS 调用原生（AndroidInterface.postMessage）
    QJniObject jsInterface("com/ggt/ggtpassword/QtBridgeInterface", "()V");
    if (jsInterface.isValid()) {
        m_jsInterface = jsInterface;
        QJniObject ifaceName = QJniObject::fromString("AndroidInterface");
        m_webView.callMethod<void>(
            "addJavascriptInterface",
            "(Ljava/lang/Object;Ljava/lang/String;)V",
            m_jsInterface.object<jobject>(),
            ifaceName.object<jstring>()
        );
        qDebug() << "AndroidWebViewBridge: addJavascriptInterface AndroidInterface installed";
    } else {
        qWarning() << "AndroidWebViewBridge: Failed to create com.ggt.ggtpassword.QtBridgeInterface";
    }

    // 将WebView添加到Activity的内容视图并设为全屏
    jint contentId = QJniObject::getStaticField<jint>("android/R$id", "content");
    QJniObject contentView = m_activity.callObjectMethod("findViewById", "(I)Landroid/view/View;", contentId);
    if (contentView.isValid()) {
        qDebug() << "AndroidWebViewBridge: Found android.R.id.content";
        // MATCH_PARENT = -1
        QJniObject layoutParams("android/widget/FrameLayout$LayoutParams", "(II)V", -1, -1);
        if (layoutParams.isValid()) {
            m_webView.callMethod<void>("setLayoutParams", "(Landroid/view/ViewGroup$LayoutParams;)V",
                                       layoutParams.object<jobject>());
        }
        // 使用 Activity.addContentView 保证添加顺序在最顶层
        qDebug() << "AndroidWebViewBridge: About to call Activity.addContentView";
        m_activity.callMethod<void>("addContentView", "(Landroid/view/View;Landroid/view/ViewGroup$LayoutParams;)V",
                                    m_webView.object<jobject>(), layoutParams.object<jobject>());
        qDebug() << "AndroidWebViewBridge: Activity.addContentView returned";

        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: Exception during addContentView";
        }

        // 提升层级并确保可见（逐步记录每个调用）
        qDebug() << "AndroidWebViewBridge: [UI] Visibility/Z begin";
        qDebug() << "AndroidWebViewBridge: setVisibility(VISIBLE) begin";
        m_webView.callMethod<void>("setVisibility", "(I)V", 0);
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in setVisibility"; }
        qDebug() << "AndroidWebViewBridge: setVisibility(VISIBLE) done";

        qDebug() << "AndroidWebViewBridge: bringToFront begin";
        m_webView.callMethod<void>("bringToFront");
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in bringToFront"; }
        qDebug() << "AndroidWebViewBridge: bringToFront done";

        qDebug() << "AndroidWebViewBridge: setZ(100.0f) begin";
        m_webView.callMethod<void>("setZ", "(F)V", 100.0f);
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in setZ"; }
        qDebug() << "AndroidWebViewBridge: setZ(100.0f) done";

        qDebug() << "AndroidWebViewBridge: setAlpha(1.0f) begin";
        m_webView.callMethod<void>("setAlpha", "(F)V", 1.0f);
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in setAlpha"; }
        qDebug() << "AndroidWebViewBridge: setAlpha(1.0f) done";

        qDebug() << "AndroidWebViewBridge: setLayerType(HARDWARE) begin";
        m_webView.callMethod<void>("setLayerType", "(ILandroid/graphics/Paint;)V", 2, nullptr);
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in setLayerType"; }
        qDebug() << "AndroidWebViewBridge: setLayerType(HARDWARE) done";

        qDebug() << "AndroidWebViewBridge: setBackgroundColor(0xFFFFFFFF) begin";
        m_webView.callMethod<void>("setBackgroundColor", "(I)V", 0xFFFFFFFF);
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in setBackgroundColor"; }
        qDebug() << "AndroidWebViewBridge: setBackgroundColor(0xFFFFFFFF) done";

        qDebug() << "AndroidWebViewBridge: setFocusable(true) begin";
        m_webView.callMethod<void>("setFocusable", "(Z)V", true);
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in setFocusable"; }
        qDebug() << "AndroidWebViewBridge: setFocusable(true) done";

        qDebug() << "AndroidWebViewBridge: setFocusableInTouchMode(true) begin";
        m_webView.callMethod<void>("setFocusableInTouchMode", "(Z)V", true);
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in setFocusableInTouchMode"; }
        qDebug() << "AndroidWebViewBridge: setFocusableInTouchMode(true) done";

        qDebug() << "AndroidWebViewBridge: requestFocus skipped to avoid hang";
        // m_webView.callMethod<void>("requestFocus");

        qDebug() << "AndroidWebViewBridge: requestLayout begin";
        m_webView.callMethod<void>("requestLayout");
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in requestLayout"; }
        qDebug() << "AndroidWebViewBridge: requestLayout done";

        qDebug() << "AndroidWebViewBridge: invalidate begin";
        m_webView.callMethod<void>("invalidate");
        if (env->ExceptionCheck()) { env->ExceptionDescribe(); env->ExceptionClear(); qWarning() << "AndroidWebViewBridge: Exception in invalidate"; }
        qDebug() << "AndroidWebViewBridge: invalidate done";

        qDebug() << "AndroidWebViewBridge: [UI] Visibility/Z end";
        // 检查可见性/层级调整后的异常
        if (env->ExceptionCheck()) {
            env->ExceptionDescribe();
            env->ExceptionClear();
            qWarning() << "AndroidWebViewBridge: Exception after visibility/Z/layout adjustments";
        }
        qDebug() << "AndroidWebViewBridge: WebView added via Activity.addContentView and brought to front";
    } else {
        qWarning() << "AndroidWebViewBridge: Failed to get content view (android.R.id.content), trying DecorView fallback";
        // DecorView fallback: add as child directly to DecorView (FrameLayout)
        QJniObject window = m_activity.callObjectMethod("getWindow", "()Landroid/view/Window;");
        if (window.isValid()) {
            QJniObject decorView = window.callObjectMethod("getDecorView", "()Landroid/view/View;");
            if (decorView.isValid()) {
                qDebug() << "AndroidWebViewBridge: Using DecorView fallback";
                // MATCH_PARENT = -1
                QJniObject layoutParams("android/view/ViewGroup$LayoutParams", "(II)V", -1, -1);
                if (layoutParams.isValid()) {
                    // addView(View, LayoutParams) 确保全屏尺寸
                    decorView.callMethod<void>(
                        "addView",
                        "(Landroid/view/View;Landroid/view/ViewGroup$LayoutParams;)V",
                        m_webView.object<jobject>(),
                        layoutParams.object<jobject>()
                    );
                    m_webView.callMethod<void>(
                        "setLayoutParams",
                        "(Landroid/view/ViewGroup$LayoutParams;)V",
                        layoutParams.object<jobject>()
                    );
                } else {
                    // 退化为 addView(View)
                    decorView.callMethod<void>("addView", "(Landroid/view/View;)V", m_webView.object<jobject>());
                    qWarning() << "AndroidWebViewBridge: Failed to create LayoutParams, used default addView";
                }
                // 可见并置顶
                m_webView.callMethod<void>("setVisibility", "(I)V", 0);
                m_webView.callMethod<void>("bringToFront");
                m_webView.callMethod<void>("setZ", "(F)V", 100.0f);
                m_webView.callMethod<void>("setLayerType", "(ILandroid/graphics/Paint;)V", 2, nullptr); // LAYER_TYPE_HARDWARE
                m_webView.callMethod<void>("setBackgroundColor", "(I)V", -1);
                // 应用布局
                m_webView.callMethod<void>("requestLayout");
                m_webView.callMethod<void>("invalidate");
                qDebug() << "AndroidWebViewBridge: WebView added to DecorView with MATCH_PARENT LayoutParams and elevated";
            } else {
                qWarning() << "AndroidWebViewBridge: DecorView invalid";
            }
        }
    }

    return true;
}

QJniObject AndroidWebViewBridge::getWebViewObject() const
{
    return m_webView;
}

void AndroidWebViewBridge::setParentActivity(const QJniObject& activity)
{
    m_activity = activity;
}

void AndroidWebViewBridge::resizeWebView(int x, int y, int width, int height)
{
    m_webViewX = x;
    m_webViewY = y;
    m_webViewWidth = width;
    m_webViewHeight = height;

    if (m_isInitialized && m_webView.isValid()) {
        QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this, x, y, width, height]() {
            m_webView.callMethod<void>("layout", "(IIII)V", x, y, x + width, y + height);
        });
    }
}

void AndroidWebViewBridge::onApplicationStateChanged(Qt::ApplicationState state)
{
    if (!m_webView.isValid()) {
        return;
    }

    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this, state]() {
        if (state == Qt::ApplicationActive) {
            m_webView.callMethod<void>("onResume");
        } else if (state == Qt::ApplicationInactive) {
            m_webView.callMethod<void>("onPause");
        }
    });
}

bool AndroidWebViewBridge::initializeJNI()
{
    m_activity = QJniObject::callStaticObjectMethod("org/qtproject/qt/android/QtNative",
                                                    "activity",
                                                    "()Landroid/app/Activity;");
    if (!m_activity.isValid()) {
        qWarning() << "AndroidWebViewBridge: Failed to get Android activity";
        return false;
    }
    return true;
}

void AndroidWebViewBridge::setupWebViewSettings()
{
    if (!m_webView.isValid()) {
        qWarning() << "AndroidWebViewBridge: setupWebViewSettings() skipped, webView invalid";
        return;
    }

    // 获取WebSettings
    QJniObject settings = m_webView.callObjectMethod("getSettings", "()Landroid/webkit/WebSettings;");
    if (!settings.isValid()) {
        qWarning() << "AndroidWebViewBridge: Failed to get WebSettings";
        return;
    }

    // 启用JavaScript
    settings.callMethod<void>("setJavaScriptEnabled", "(Z)V", true);
    // 启用DOM存储
    settings.callMethod<void>("setDomStorageEnabled", "(Z)V", true);
    // 启用数据库
    settings.callMethod<void>("setDatabaseEnabled", "(Z)V", true);

    // 设置用户代理
    QString userAgent = "GGTPassword Android WebView";
    QJniEnvironment env;
    jstring jUserAgent = env->NewStringUTF(userAgent.toUtf8().constData());
    settings.callMethod<void>("setUserAgentString", "(Ljava/lang/String;)V", jUserAgent);
    env->DeleteLocalRef(jUserAgent);

    // 允许文件访问
    settings.callMethod<void>("setAllowFileAccess", "(Z)V", true);
    // 允许内容访问
    settings.callMethod<void>("setAllowContentAccess", "(Z)V", true);
    // 允许 file:// → file:// 的访问（Android 4.1+，仍建议设置）
    settings.callMethod<void>("setAllowFileAccessFromFileURLs", "(Z)V", true);
    // 允许 file:// → http(s) 的跨源访问（加载本地 HTML 中外链资源常用）
    settings.callMethod<void>("setAllowUniversalAccessFromFileURLs", "(Z)V", true);

    // 设置缓存模式
    settings.callMethod<void>("setCacheMode", "(I)V", -1);

    qDebug() << "AndroidWebViewBridge: WebSettings applied (JS/DOM/DB/file access/universal)";
}

void AndroidWebViewBridge::injectBridgeScript()
{
    QString bridgeScript = R"(
        (function() {
            if (window.__QtBridgeInjected) { return; }
            window.__QtBridgeInjected = true;

            var QtBridge = {
                sendMessage: function(method, params) {
                    var msg = { method: method, params: params || {} };
                    if (window.AndroidInterface && typeof window.AndroidInterface.postMessage === 'function') {
                        try { window.AndroidInterface.postMessage(JSON.stringify(msg)); }
                        catch (e) { console.log('[Android] postMessage error:', e); }
                    } else {
                        console.log('[Android] AndroidInterface missing, message:', msg);
                    }
                },
                postMessage: function(message) {
                    if (window.AndroidInterface && typeof window.AndroidInterface.postMessage === 'function') {
                        try { window.AndroidInterface.postMessage(message); }
                        catch (e) { console.log('[Android] postMessage error:', e); }
                    } else {
                        console.log('[Android] AndroidInterface missing, raw:', message);
                    }
                },
                // 供页面注册原生→网页消息回调
                onMessage: function(callback) {
                    if (typeof callback === 'function') {
                        window.nativeMessageCallback = callback;
                        console.log('[Android] onMessage callback registered');
                    } else {
                        console.log('[Android] onMessage requires a function');
                    }
                },
                // 原生调用此方法把消息喂给页面
                handleMessage: function(message) {
                    try {
                        var data = (typeof message === 'string') ? JSON.parse(message) : message;
                        if (window.nativeMessageCallback && typeof window.nativeMessageCallback === 'function') {
                            window.nativeMessageCallback(data);
                        } else {
                            console.log('[Android] nativeMessageCallback not set:', data);
                        }
                    } catch (e) {
                        console.log('[Android] handleMessage JSON parse error:', e, message);
                    }
                }
            };

            window.QtBridge = QtBridge;
            window.NativeBridge = QtBridge; // 兼容其他平台的命名
            console.log('[Android] QtBridge injected');

            // Debug: 注入错误与控制台远程日志（可选）
            (function(){
                if (!QtBridge) { return; }

                var remoteUrl = null;
                var consoleForwardEnabled = false;
                function setRemote(url){
                    remoteUrl = url || null;
                    console.log('[Android] Remote log URL set:', remoteUrl);
                    if (remoteUrl && !consoleForwardEnabled) {
                        enableConsoleForwarding();
                    }
                }
                function sendRemote(type, payload){
                    if (!remoteUrl) { return; }
                    try {
                        fetch(remoteUrl, {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify(Object.assign({ type: type }, payload || {}))
                        });
                    } catch (e) {
                        console.log('[Android] remote log failed:', e);
                    }
                }

                // 扩展 handleMessage 以支持设置远程日志 URL
                var _origHandle = QtBridge.handleMessage;
                QtBridge.handleMessage = function(message){
                    try {
                        var data = (typeof message === 'string') ? JSON.parse(message) : message;
                        if (data && data.method === 'setRemoteLogUrl') {
                            setRemote(data.params && data.params.url);
                            return;
                        }
                    } catch (e) { /* 忽略 */ }
                    return _origHandle.call(QtBridge, message);
                };

                // 捕获 JS 错误
                window.onerror = function(msg, src, line, col, err){
                    var payload = {
                        msg: String(msg),
                        src: String(src || ''),
                        line: Number(line || 0),
                        col: Number(col || 0),
                        stack: err && err.stack ? String(err.stack) : null,
                        url: String(document.location.href),
                        ua: String(navigator.userAgent),
                        ts: Date.now()
                    };
                    try { console.error('[JS Error]', msg, src + ':' + line + ':' + col, payload.stack || ''); } catch (_) {}
                    sendRemote('error', payload);
                    return false;
                };

                // 捕获未处理的 Promise 拒绝
                window.addEventListener('unhandledrejection', function(ev){
                    var reason = ev && ev.reason;
                    var payload = {
                        msg: reason ? String(reason.message || reason) : 'unhandledrejection',
                        src: '',
                        line: 0,
                        col: 0,
                        stack: reason && reason.stack ? String(reason.stack) : null,
                        url: String(document.location.href),
                        ua: String(navigator.userAgent),
                        ts: Date.now()
                    };
                    try { console.error('[Unhandled Rejection]', payload.msg, payload.stack || ''); } catch (_) {}
                    sendRemote('unhandledrejection', payload);
                });

                // 远程转发 console（不影响 WebChromeClient 的 Logcat 输出），仅在设置了远程URL后启用
                function enableConsoleForwarding(){
                    if (consoleForwardEnabled) { return; }
                    consoleForwardEnabled = true;
                    var orig = {
                        log: console.log && console.log.bind(console),
                        info: console.info ? console.info.bind(console) : console.log.bind(console),
                        warn: console.warn ? console.warn.bind(console) : console.log.bind(console),
                        error: console.error ? console.error.bind(console) : console.log.bind(console)
                    };
                    function forward(level, args){
                        var msg = Array.prototype.slice.call(args).map(function(a){
                            try { return (typeof a === 'string') ? a : JSON.stringify(a); }
                            catch(_) { return String(a); }
                        }).join(' ');
                        sendRemote('console', {
                            level: level,
                            msg: msg,
                            url: String(document.location.href),
                            ua: String(navigator.userAgent),
                            ts: Date.now()
                        });
                    }
                    console.log = function(){ forward('log', arguments); return orig.log.apply(console, arguments); };
                    console.info = function(){ forward('info', arguments); return orig.info.apply(console, arguments); };
                    console.warn = function(){ forward('warn', arguments); return orig.warn.apply(console, arguments); };
                    console.error = function(){ forward('error', arguments); return orig.error.apply(console, arguments); };
                }
            })();

        })();
    )";

    qDebug() << "AndroidWebViewBridge: injectBridgeScript length:" << bridgeScript.length();
    evaluateJavaScript(bridgeScript);
    qDebug() << "AndroidWebViewBridge: injectBridgeScript executed";
}

void AndroidWebViewBridge::sendMessageToWeb(const QString& method, const QJsonObject& params)
{
    QJsonObject message;
    message["method"] = method;
    message["params"] = params;
    
    QJsonDocument doc(message);
    QString jsonString = doc.toJson(QJsonDocument::Compact);
    
    QString script = QString("if (window.QtBridge && window.QtBridge.handleMessage) { "
                            "window.QtBridge.handleMessage(%1); }").arg(jsonString);

    QString jsonPreview = jsonString.size() > 300 ? (jsonString.left(300) + "...") : jsonString;
    qDebug() << "AndroidWebViewBridge: sendMessageToWeb() method:" << method << ", json length:" << jsonString.length() << ", preview:" << jsonPreview;
    
    evaluateJavaScript(script);
}

void AndroidWebViewBridge::handleWebViewMessage(const QString& message)
{
    qDebug() << "AndroidWebViewBridge: handleWebViewMessage received:" << message;
    handleMessage(message);
}

void AndroidWebViewBridge::cleanup()
{
    // 销毁WebView需在UI线程执行
    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this]() {
        if (m_webView.isValid()) {
            m_webView.callMethod<void>("destroy");
        }
    }).waitForFinished();
    
    m_webView = QJniObject();
    m_activity = QJniObject();
    m_webViewClient = QJniObject();
    m_webChromeClient = QJniObject();
    m_jsInterface = QJniObject();
    
    m_isInitialized = false;
}

// JNI回调函数实现
extern "C" {
    JNIEXPORT void JNICALL
    Java_org_qtproject_qt_android_bindings_QtActivity_onWebViewMessage(
        JNIEnv *env, jobject thiz, jstring message)
    {
        if (g_androidWebViewBridge) {
            const char* messageStr = env->GetStringUTFChars(message, nullptr);
            QString qMessage = QString::fromUtf8(messageStr);
            env->ReleaseStringUTFChars(message, messageStr);
            
            QMetaObject::invokeMethod(g_androidWebViewBridge, "handleWebViewMessage",
                                      Qt::QueuedConnection, Q_ARG(QString, qMessage));
        }
    }
    
    JNIEXPORT void JNICALL
    Java_org_qtproject_qt_android_bindings_QtActivity_onWebViewPageFinished(
        JNIEnv *env, jobject thiz, jstring url)
    {
        if (g_androidWebViewBridge) {
            const char* urlStr = env->GetStringUTFChars(url, nullptr);
            QString qUrl = QString::fromUtf8(urlStr);
            env->ReleaseStringUTFChars(url, urlStr);
            
            QMetaObject::invokeMethod(g_androidWebViewBridge, "handleWebViewMessage",
                                      Qt::QueuedConnection, Q_ARG(QString, QString("pageFinished:" + qUrl)));
        }
    }
    
    JNIEXPORT void JNICALL
    Java_org_qtproject_qt_android_bindings_QtActivity_onWebViewPageStarted(
        JNIEnv *env, jobject thiz, jstring url)
    {
        if (g_androidWebViewBridge) {
            const char* urlStr = env->GetStringUTFChars(url, nullptr);
            QString qUrl = QString::fromUtf8(urlStr);
            env->ReleaseStringUTFChars(url, urlStr);
            
            QMetaObject::invokeMethod(g_androidWebViewBridge, "handleWebViewMessage",
                                      Qt::QueuedConnection, Q_ARG(QString, QString("pageStarted:" + qUrl)));
        }
    }
}

extern "C" JNIEXPORT void JNICALL
Java_com_ggt_ggtpassword_QtBridgeInterface_onMessage(JNIEnv* env, jclass /*clazz*/, jstring message)
{
    const char* cmsg = env->GetStringUTFChars(message, nullptr);
    QString msg = QString::fromUtf8(cmsg ? cmsg : "");
    env->ReleaseStringUTFChars(message, cmsg);

    if (g_androidWebViewBridge) {
        g_androidWebViewBridge->handleWebViewMessage(msg);
    } else {
        qWarning() << "AndroidWebViewBridge: g_androidWebViewBridge is null in QtBridgeInterface.onMessage";
    }
}

#endif // Q_OS_ANDROID