#ifndef ANDROIDVIEW_H
#define ANDROIDVIEW_H

#include "../../mvc/BaseView.h"

#ifdef Q_OS_ANDROID

#include <QObject>
#include <QJniObject>

class AndroidWebViewBridge;

/**
 * @brief Android平台的视图管理类
 * 负责管理Android平台的UI和WebView集成
 */
class AndroidView : public BaseView
{
    Q_OBJECT

public:
    explicit AndroidView(QWidget* parent = nullptr);
    ~AndroidView() override;

    // 实现基类的纯虚函数
    WebViewBridge* getWebViewBridge() override;
    void initializeView() override;
    void loadHtml(const QString& html) override;
    void loadUrl(const QString& url) override;
    void showView() override;

    /**
     * @brief 设置父Activity
     * @param activity Activity实例
     */
    void setParentActivity(const QJniObject& activity);
    
    /**
     * @brief 获取Activity实例
     * @return QJniObject Activity实例
     */
    QJniObject getActivity() const;

    /**
     * @brief 设置全屏模式
     * @param fullscreen 是否全屏
     */
    void setFullScreen(bool fullscreen);

    /**
     * @brief 设置状态栏可见性
     * @param visible 是否可见
     */
    void setStatusBarVisible(bool visible);

    /**
     * @brief 设置导航栏可见性
     * @param visible 是否可见
     */
    void setNavigationBarVisible(bool visible);

    /**
     * @brief 设置屏幕方向
     * @param orientation 屏幕方向 (1=portrait, 2=landscape)
     */
    void setScreenOrientation(int orientation);

    /**
     * @brief 请求权限
     * @param permission 权限名称
     * @return 是否成功请求
     */
    bool requestPermission(const QString& permission);

    /**
     * @brief 检查权限状态
     * @param permission 权限名称
     * @return 是否已授权
     */
    bool checkPermission(const QString& permission);

    // 生命周期方法
    void onResume();
    void onPause();
    void onDestroy();

    // 窗口事件处理
    void onWindowFocusChanged(bool hasFocus);
    void onConfigurationChanged();

public slots:
    /**
     * @brief 处理Activity生命周期事件
     */
    void onActivityCreated();
    void onActivityStarted();
    void onActivityResumed();
    void onActivityPaused();
    void onActivityStopped();
    void onActivityDestroyed();

    /**
     * @brief 处理系统返回键
     * @return 是否处理了返回键事件
     */
    bool onBackPressed();

    /**
     * @brief 处理权限请求结果
     * @param requestCode 请求码
     * @param permissions 权限列表
     * @param grantResults 授权结果
     */
    void onRequestPermissionsResult(int requestCode, const QStringList& permissions, 
                                   const QList<int>& grantResults);

private slots:
    void onWebViewReady();
    void applyPostInitUi();

signals:
    /**
     * @brief Activity准备就绪信号
     */
    void activityReady();

    /**
     * @brief 返回键按下信号
     */
    void backPressed();

    /**
     * @brief 权限请求结果信号
     */
    void permissionResult(int requestCode, const QString& permission, bool granted);

protected:
    void resizeEvent(QResizeEvent* event) override;

private:
    void setupLayout();
    void initializeJNI();
    void setupWebView();

    /**
     * @brief 初始化Activity设置
     */
    void initializeActivity();

    /**
     * @brief 设置Activity主题和样式
     */
    void setupActivityTheme();

    /**
     * @brief 请求必要的权限
     */
    void requestPermissions();

    /**
     * @brief 获取屏幕尺寸
     */
    void updateScreenSize();

    /**
     * @brief 应用系统UI设置
     */
    void applySystemUiSettings();

    // JNI相关方法
    void callJavaMethod(const QString& methodName, const QString& signature = "()V");
    void callJavaMethodWithParams(const QString& methodName, const QString& signature, 
                                 const QJniObject& params);

private:
    AndroidWebViewBridge* m_webViewBridge;  // WebView桥接对象
    QJniObject m_activity;           // Activity对象
    QJniObject m_window;             // Window对象
    QJniObject m_decorView;          // DecorView对象
    QJniObject m_webViewHelper;
    
    bool m_isInitialized;                   // 是否已初始化
    bool m_isFullScreen;                    // 是否全屏
    bool m_statusBarVisible;                // 状态栏是否可见
    bool m_navigationBarVisible;            // 导航栏是否可见
    bool m_themeApplied;                    // 主题/窗口标志是否已应用
    
    int m_screenWidth;                      // 屏幕宽度
    int m_screenHeight;                     // 屏幕高度
    int m_webViewX;                         // WebView X坐标
    int m_webViewY;                         // WebView Y坐标
    int m_webViewWidth;                     // WebView宽度
    int m_webViewHeight;                    // WebView高度
    
    QString m_activityTitle;                // Activity标题
    int m_screenOrientation;                // 屏幕方向
};

#endif // Q_OS_ANDROID

#endif // ANDROIDVIEW_H