#include "AndroidView.h"

#ifdef Q_OS_ANDROID

#include "AndroidWebViewBridge.h"
#include <QDebug>
#include <QResizeEvent>
#include <QJniEnvironment>
#include <QtCore/qjniobject.h>
#include <QTimer>
#include <QJsonObject>
#if WEBVIEW_DEBUG_MODE
#include "debug/DebugManager.h"
#endif

AndroidView::AndroidView(QWidget* parent)
    : BaseView(parent)
    , m_webViewBridge(nullptr)
    , m_isInitialized(false)
    , m_isFullScreen(false)
    , m_statusBarVisible(true)
    , m_navigationBarVisible(true)
    , m_screenWidth(800)
    , m_screenHeight(600)
    , m_webViewX(0)
    , m_webViewY(0)
    , m_webViewWidth(800)
    , m_webViewHeight(600)
    , m_screenOrientation(1) // Portrait
{
}

AndroidView::~AndroidView()
{
    if (m_webViewBridge) {
        delete m_webViewBridge;
        m_webViewBridge = nullptr;
    }
}

void AndroidView::initializeView()
{
    if (m_isInitialized) {
        qDebug() << "AndroidView: initializeView() called but already initialized";
        return;
    }

    qDebug() << "AndroidView: initializeView() starting";

    // 初始化JNI
    initializeJNI();

    // 设置Activity相关
    initializeActivity();
    qDebug() << "AndroidView: Activity and Window initialized";

    // 延迟应用窗口/UI设置，避免在启动期触碰视图
    QTimer::singleShot(0, this, &AndroidView::applyPostInitUi);

    // 创建WebView桥接
    m_webViewBridge = new AndroidWebViewBridge(this);
    m_webViewBridge->setParentActivity(m_activity);
    qDebug() << "AndroidView: AndroidWebViewBridge created and parent Activity set";

    connect(m_webViewBridge, &AndroidWebViewBridge::webViewReady, this, &AndroidView::onWebViewReady);

    // 初始化WebView
    qDebug() << "AndroidView: setupWebView()";
    setupWebView();
    qDebug() << "AndroidView: setupWebView() dispatched";

    // 更新屏幕尺寸并设置默认WebView大小
    updateScreenSize();
    qDebug() << "AndroidView: updateScreenSize() applied, size:" << m_screenWidth << "x" << m_screenHeight;

    // 兜底：初始化阶段直接加载 assets 页面，避免 webViewReady 信号缺失导致空白
    {
        const QString assetUrl = QStringLiteral("file:///android_asset/web/index.html");
        qDebug() << "AndroidView: initializeView() initial asset load URL:" << assetUrl;
        m_webViewBridge->loadUrl(assetUrl);
    }

    // 请求必要权限
    requestPermissions();
    qDebug() << "AndroidView: requestPermissions() called";

    m_isInitialized = true;
    qDebug() << "AndroidView: initializeView() completed";
}

void AndroidView::applyPostInitUi()
{
    // 只在首次初始化后应用
    if (!m_themeApplied) {
        setupActivityTheme();
        applySystemUiSettings();
        m_themeApplied = true;
        qDebug() << "AndroidView: Post-init UI applied";
    }
}

void AndroidView::showView()
{
    if (!m_isInitialized) {
        initializeView();
    }
}

WebViewBridge* AndroidView::getWebViewBridge()
{
    return m_webViewBridge;
}

void AndroidView::setParentActivity(const QJniObject& activity)
{
    m_activity = activity;
}

QJniObject AndroidView::getActivity() const
{
    return m_activity;
}

void AndroidView::setFullScreen(bool fullscreen)
{
    m_isFullScreen = fullscreen;
    applySystemUiSettings();
}

void AndroidView::setStatusBarVisible(bool visible)
{
    m_statusBarVisible = visible;
    applySystemUiSettings();
}

void AndroidView::setNavigationBarVisible(bool visible)
{
    m_navigationBarVisible = visible;
    applySystemUiSettings();
}

void AndroidView::setScreenOrientation(int orientation)
{
    m_screenOrientation = orientation;
}

void AndroidView::onActivityCreated()
{
    qDebug() << "AndroidView: Activity created";
}

void AndroidView::onActivityStarted()
{
    qDebug() << "AndroidView: Activity started";
}

void AndroidView::onActivityResumed()
{
    qDebug() << "AndroidView: Activity resumed";

    if (!m_themeApplied) {
        setupActivityTheme();
        applySystemUiSettings();
        m_themeApplied = true;
    }

    // Activity 恢复后再次确保 WebView 位于最顶层且可见
    if (m_webViewBridge) {
        AndroidWebViewBridge* bridge = static_cast<AndroidWebViewBridge*>(m_webViewBridge);
        QJniObject webView = bridge->getWebViewObject();
        if (webView.isValid()) {
            QNativeInterface::QAndroidApplication::runOnAndroidMainThread([webView]() {
                webView.callMethod<void>("bringToFront");
                webView.callMethod<void>("setZ", "(F)V", 100.0f);
                webView.callMethod<void>("setVisibility", "(I)V", 0); // View.VISIBLE
                webView.callMethod<void>("requestLayout");
                webView.callMethod<void>("invalidate");
                webView.callMethod<void>("onResume");
            });
            qDebug() << "AndroidView: WebView brought to front on resume";
        }
    }
}

void AndroidView::onActivityPaused()
{
    qDebug() << "AndroidView: Activity paused";

    if (m_webViewBridge && m_webViewBridge->getWebViewObject().isValid()) {
        QJniObject webView = m_webViewBridge->getWebViewObject();
        QNativeInterface::QAndroidApplication::runOnAndroidMainThread([webView]() {
            webView.callMethod<void>("onPause");
        });
    }
}

void AndroidView::onActivityStopped()
{
    qDebug() << "AndroidView: Activity stopped";
}

void AndroidView::onActivityDestroyed()
{
    qDebug() << "AndroidView: Activity destroyed";
}

bool AndroidView::onBackPressed()
{
    bool handled = false;

    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([&]() {
        if (m_webViewBridge && m_webViewBridge->getWebViewObject().isValid()) {
            QJniObject webView = m_webViewBridge->getWebViewObject();
            jboolean canGoBack = webView.callMethod<jboolean>("canGoBack");
            if (canGoBack) {
                webView.callMethod<void>("goBack");
                handled = true;
            }
        }
    }).waitForFinished();

    if (handled) {
        return true; // 已处理返回键
    }

    emit backPressed();
    return false; // 未处理，让系统处理
}

void AndroidView::onRequestPermissionsResult(int requestCode, const QStringList& permissions, 
                                           const QList<int>& grantResults)
{
    for (int i = 0; i < permissions.size() && i < grantResults.size(); ++i) {
        bool granted = (grantResults[i] == 0); // PackageManager.PERMISSION_GRANTED = 0
        emit permissionResult(requestCode, permissions[i], granted);
        
        qDebug() << "AndroidView: Permission" << permissions[i] 
                 << (granted ? "granted" : "denied");
    }
}

void AndroidView::onWebViewReady()
{
    qDebug() << "AndroidView: WebView ready";

    if (m_webViewBridge) {
        qDebug() << "AndroidView: resizeWebView to" << m_webViewWidth << "x" << m_webViewHeight
                 << "at" << m_webViewX << "," << m_webViewY;
        m_webViewBridge->resizeWebView(m_webViewX, m_webViewY, m_webViewWidth, m_webViewHeight);

        QJniObject webView = m_webViewBridge->getWebViewObject();
        qDebug() << "AndroidView: WebView object valid:" << webView.isValid();

#if WEBVIEW_DEBUG_MODE
        // 在调试模式下，如果环境变量提供了远程日志地址，则下发到页面
        if (DebugManager::instance()->isDebugEnabled()) {
            QByteArray remoteUrlEnv = qgetenv("WEBVIEW_REMOTE_LOG_URL");
            if (!remoteUrlEnv.isEmpty()) {
                const QString remoteUrl = QString::fromUtf8(remoteUrlEnv);
                QJsonObject params; params["url"] = remoteUrl;
                m_webViewBridge->sendMessage("setRemoteLogUrl", params);
                qDebug() << "AndroidView: setRemoteLogUrl sent to WebView:" << remoteUrl;
            }
        }
#endif

        const QString assetUrl = QStringLiteral("file:///android_asset/web/index.html");
        qDebug() << "AndroidView: onWebViewReady() fallback load asset URL:" << assetUrl;
        m_webViewBridge->loadUrl(assetUrl);
    } else {
        qWarning() << "AndroidView: onWebViewReady() but m_webViewBridge is null";
    }
}

void AndroidView::initializeActivity()
{
    if (!m_activity.isValid()) {
        qWarning() << "AndroidView: initializeActivity() skipped, m_activity invalid";
        return;
    }

    m_window = m_activity.callObjectMethod("getWindow", "()Landroid/view/Window;");
    qDebug() << "AndroidView: Window obtained:" << m_window.isValid();

    if (m_window.isValid()) {
        m_decorView = m_window.callObjectMethod("getDecorView", "()Landroid/view/View;");
        qDebug() << "AndroidView: DecorView obtained:" << m_decorView.isValid();
    } else {
        qWarning() << "AndroidView: Window invalid, DecorView not retrieved";
    }

    if (m_webViewBridge) {
        m_webViewBridge->setParentActivity(m_activity);
        qDebug() << "AndroidView: setParentActivity() applied to WebViewBridge";
    } else {
        qWarning() << "AndroidView: WebViewBridge null during initializeActivity()";
    }
}

void AndroidView::setupActivityTheme()
{
    if (!m_window.isValid()) {
        return;
    }
    
    // Manifest 已开启硬件加速；避免启动期 addFlags 触摸视图
    // 如需额外标志（如保持常亮），请在 Activity 恢复后调用
    // QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this]() {
    //     m_window.callMethod<void>("addFlags", "(I)V", 0x00000080 /*FLAG_KEEP_SCREEN_ON*/);
    // }).waitForFinished();
}

void AndroidView::setupWebView()
{
    if (m_webViewBridge) {
        m_webViewBridge->initializeWebView();
        qDebug() << "AndroidView: initializeWebView() called";
    } else {
        qWarning() << "AndroidView: setupWebView() skipped, m_webViewBridge is null";
    }
}

void AndroidView::requestPermissions()
{
    QStringList permissions = {
        "android.permission.INTERNET",
        "android.permission.ACCESS_NETWORK_STATE",
        "android.permission.WRITE_EXTERNAL_STORAGE",
        "android.permission.READ_EXTERNAL_STORAGE"
    };
    
    if (!m_activity.isValid()) {
        qDebug() << "Activity is not valid, cannot request permissions";
        return;
    }
    
    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this, permissions]() {
        QJniEnvironment env;
        
        // 创建权限数组
        jobjectArray jPermissions = env->NewObjectArray(permissions.size(), 
                                                       env->FindClass("java/lang/String"), 
                                                       nullptr);
        
        for (int i = 0; i < permissions.size(); ++i) {
            jstring jPermission = env->NewStringUTF(permissions[i].toUtf8().constData());
            env->SetObjectArrayElement(jPermissions, i, jPermission);
            env->DeleteLocalRef(jPermission);
        }
        
        // 直接使用JNI调用Android的requestPermissions方法
        m_activity.callMethod<void>("requestPermissions", 
                                    "([Ljava/lang/String;I)V",
                                    jPermissions, 1001);
        
        env->DeleteLocalRef(jPermissions);
    });
}

bool AndroidView::requestPermission(const QString& permission)
{
    if (!m_activity.isValid()) {
        return false;
    }

    bool granted = false;

    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([&]() {
        QJniEnvironment env;
        jstring jPermission = env->NewStringUTF(permission.toUtf8().constData());
        
        // 检查权限
        jint checkResult = m_activity.callMethod<jint>(
            "checkSelfPermission",
            "(Ljava/lang/String;)I",
            jPermission
        );
        
        // PackageManager.PERMISSION_GRANTED = 0
        granted = (checkResult == 0);
        
        env->DeleteLocalRef(jPermission);
    }).waitForFinished();

    return granted;
}

bool AndroidView::checkPermission(const QString& permission)
{
    return requestPermission(permission);
}

void AndroidView::updateScreenSize()
{
    if (!m_activity.isValid()) {
        return;
    }

    int width = m_screenWidth;
    int height = m_screenHeight;

    // 在UI线程获取DecorView大小
    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([&]() {
        if (m_decorView.isValid()) {
            width = m_decorView.callMethod<jint>("getWidth");
            height = m_decorView.callMethod<jint>("getHeight");
        }
    }).waitForFinished();

    if (width > 0 && height > 0) {
        m_screenWidth = width;
        m_screenHeight = height;
    }

    // 默认WebView占满屏幕
    m_webViewWidth = m_screenWidth;
    m_webViewHeight = m_screenHeight;

    qDebug() << "AndroidView: Screen size:" << m_screenWidth << "x" << m_screenHeight;

    // 确保WebView在获得屏幕尺寸后填满父视图
    if (m_webViewBridge) {
        m_webViewBridge->resizeWebView(m_webViewX, m_webViewY, m_webViewWidth, m_webViewHeight);
    }
}

void AndroidView::applySystemUiSettings()
{
    if (!m_decorView.isValid()) {
        return;
    }
    
    int flags = 0;
    
    if (m_isFullScreen) {
        // 全屏模式标志
        flags |= 0x00000004; // SYSTEM_UI_FLAG_FULLSCREEN
        flags |= 0x00000002; // SYSTEM_UI_FLAG_HIDE_NAVIGATION
        flags |= 0x00001000; // SYSTEM_UI_FLAG_IMMERSIVE_STICKY
    }
    
    if (!m_statusBarVisible) {
        flags |= 0x00000004; // SYSTEM_UI_FLAG_FULLSCREEN
    }
    
    if (!m_navigationBarVisible) {
        flags |= 0x00000002; // SYSTEM_UI_FLAG_HIDE_NAVIGATION
    }
    
    // 应用系统UI标志需在Android UI线程执行
    QNativeInterface::QAndroidApplication::runOnAndroidMainThread([this, flags]() {
        m_decorView.callMethod<void>("setSystemUiVisibility", "(I)V", flags);
    });
    
    qDebug() << "AndroidView: Applied system UI settings, flags:" << flags;
}

void AndroidView::initializeJNI()
{
    qDebug() << "AndroidView: Initializing JNI";
    
    // 获取当前的Activity
    m_activity = QJniObject::callStaticObjectMethod("org/qtproject/qt/android/QtNative", 
                                                    "activity", 
                                                    "()Landroid/app/Activity;");
    
    if (!m_activity.isValid()) {
        qDebug() << "AndroidView: Failed to get Activity";
        return;
    }
    
    // 获取Window对象
    m_window = m_activity.callObjectMethod("getWindow", "()Landroid/view/Window;");
    if (m_window.isValid()) {
        // 获取DecorView
        m_decorView = m_window.callObjectMethod("getDecorView", "()Landroid/view/View;");
    }
    
    qDebug() << "AndroidView: JNI initialization completed";
}

void AndroidView::onDestroy()
{
    qDebug() << "AndroidView: onDestroy called";
    
    if (m_webViewBridge) {
        m_webViewBridge->cleanup();
    }
}

void AndroidView::resizeEvent(QResizeEvent* event)
{
    BaseView::resizeEvent(event);

    // 更新WebView大小
    m_webViewWidth = event->size().width();
    m_webViewHeight = event->size().height();

    if (m_webViewBridge) {
        m_webViewBridge->resizeWebView(m_webViewX, m_webViewY, m_webViewWidth, m_webViewHeight);
        qDebug() << "AndroidView: resizeEvent applied webView size:" << m_webViewWidth << "x" << m_webViewHeight
                 << "at" << m_webViewX << "," << m_webViewY;
    } else {
        qWarning() << "AndroidView: resizeEvent skipped, m_webViewBridge is null";
    }
}

void AndroidView::loadHtml(const QString& html)
{
    if (!m_webViewBridge) {
        qWarning() << "AndroidView: loadHtml() skipped, m_webViewBridge is null";
        return;
    }

    qDebug() << "AndroidView: loadHtml() received inline HTML length" << html.length()
             << ", preview:" << html.left(200);

    // 在 Android 上优先从 assets 加载，保证相对资源（bridge.js、样式等）可解析
    const QString assetUrl = QStringLiteral("file:///android_asset/web/index.html");
    qDebug() << "AndroidView: loadHtml() ignored inline HTML, loading asset URL:" << assetUrl;
    m_webViewBridge->loadUrl(assetUrl);
}

void AndroidView::loadUrl(const QString& url)
{
    if (!m_webViewBridge) {
        qWarning() << "AndroidView: loadUrl() skipped, m_webViewBridge is null";
        return;
    }

    qDebug() << "AndroidView: loadUrl() ->" << url;

    QJniObject webView = m_webViewBridge->getWebViewObject();
    qDebug() << "AndroidView: WebView object valid before loadUrl:" << webView.isValid();

    m_webViewBridge->loadUrl(url);
}

#endif // Q_OS_ANDROID