#ifndef BASECONTROLLER_H
#define BASECONTROLLER_H

#include <QObject>
#include <QJsonObject>
#include <QString>
#include "BaseModel.h"

class WebViewBridge;

/**
 * @brief MVC架构中的Controller基类
 * 负责处理业务逻辑和协调Model与View
 */
class BaseController : public QObject
{
    Q_OBJECT

public:
    explicit BaseController(QObject *parent = nullptr);
    virtual ~BaseController() = default;

    /**
     * @brief 设置WebView桥接对象
     * @param bridge WebView桥接对象
     */
    void setWebViewBridge(WebViewBridge* bridge);

    /**
     * @brief 设置模型对象
     * @param model 模型对象
     */
    void setModel(BaseModel* model);

    /**
     * @brief 处理来自WebView的消息
     * @param method 方法名
     * @param params 参数
     */
    virtual void handleWebViewMessage(const QString& method, const QJsonObject& params) = 0;

    /**
     * @brief 初始化控制器
     */
    virtual void initialize() = 0;

protected:
    /**
     * @brief 向WebView发送消息
     * @param method 方法名
     * @param params 参数
     */
    void sendToWebView(const QString& method, const QJsonObject& params = QJsonObject());

    /**
     * @brief 向WebView发送响应
     * @param method 方法名
     * @param data 数据
     * @param success 是否成功
     * @param error 错误信息
     */
    void sendResponse(const QString& method, const QJsonObject& data = QJsonObject(), 
                     bool success = true, const QString& error = QString());

    /**
     * @brief 获取模型对象
     * @return 模型对象指针
     */
    BaseModel* getModel() const;

    /**
     * @brief 获取WebView桥接对象
     * @return WebView桥接对象指针
     */
    WebViewBridge* getWebViewBridge() const;

protected slots:
    /**
     * @brief 处理模型数据变化
     */
    virtual void onModelDataChanged();

    /**
     * @brief 处理模型状态变化
     * @param status 新状态
     */
    virtual void onModelStatusChanged(const QString& status);

private:
    WebViewBridge* m_webViewBridge;
    BaseModel* m_model;
};

#endif // BASECONTROLLER_H