#include "BaseController.h"
#include "../bridge/WebViewBridge.h"
#include <QDebug>

BaseController::BaseController(QObject *parent)
    : QObject(parent)
    , m_webViewBridge(nullptr)
    , m_model(nullptr)
{
}

void BaseController::setWebViewBridge(WebViewBridge* bridge)
{
    m_webViewBridge = bridge;
    
    if (m_webViewBridge) {
        // 设置消息处理器
        m_webViewBridge->setMessageHandler([this](const QString& method, const QJsonObject& params) {
            this->handleWebViewMessage(method, params);
        });
        
        // 连接信号
        connect(m_webViewBridge, &WebViewBridge::messageReceived,
                this, &BaseController::handleWebViewMessage);
    }
}

void BaseController::setModel(BaseModel* model)
{
    // 断开旧模型的连接
    if (m_model) {
        disconnect(m_model, nullptr, this, nullptr);
    }
    
    m_model = model;
    
    // 连接新模型的信号
    if (m_model) {
        connect(m_model, &BaseModel::dataChanged,
                this, &BaseController::onModelDataChanged);
        connect(m_model, &BaseModel::statusChanged,
                this, &BaseController::onModelStatusChanged);
    }
}

void BaseController::sendToWebView(const QString& method, const QJsonObject& params)
{
    if (m_webViewBridge) {
        m_webViewBridge->sendMessage(method, params);
    } else {
        qWarning() << "WebViewBridge not set, cannot send message:" << method;
    }
}

void BaseController::sendResponse(const QString& method, const QJsonObject& data, 
                                 bool success, const QString& error)
{
    if (m_webViewBridge) {
        QJsonObject response;
        response["method"] = method;
        response["success"] = success;
        response["data"] = data;
        
        if (!error.isEmpty()) {
            response["error"] = error;
        }
        
        m_webViewBridge->sendMessage("response", response);
    }
}

BaseModel* BaseController::getModel() const
{
    return m_model;
}

WebViewBridge* BaseController::getWebViewBridge() const
{
    return m_webViewBridge;
}

void BaseController::onModelDataChanged()
{
    // 默认实现：通知WebView数据已更改
    if (m_model) {
        sendToWebView("dataChanged", m_model->toJson());
    }
}

void BaseController::onModelStatusChanged(const QString& status)
{
    // 默认实现：通知WebView状态已更改
    QJsonObject params;
    params["status"] = status;
    sendToWebView("statusChanged", params);
}