#ifdef Q_OS_IOS
// iOS平台只需要UIKit和AppDelegate，完全不包含Qt头文件
#import <UIKit/UIKit.h>
#import "platform/ios/AppDelegate.h"

// 确保不链接Qt的main函数
#define QT_NO_LINK_QTMAIN 1

#else
// 非iOS平台的Qt相关头文件
#include "mainwindow.h"
#include "demo/DemoModel.h"
#include "demo/DemoController.h"

#ifdef Q_OS_MACOS
#include "platform/macos/MacOSView.h"
#endif

#ifdef Q_OS_WIN
#include "platform/windows/WindowsView.h"
#include <windows.h>
#include <io.h>
#include <fcntl.h>
#include <stdio.h>
#include <QTextStream>
#endif

#ifdef Q_OS_ANDROID
#include "platform/android/AndroidView.h"
#endif

#include <QApplication>
#include <QMainWindow>
#include <QVBoxLayout>
#include <QWidget>
#include <QFile>
#include <QIODevice>
#include <QTimer>
#include <QCommandLineParser>
#include <QCommandLineOption>
#include <QDebug>

#if WEBVIEW_DEBUG_MODE
#include "debug/DebugManager.h"
#endif
#endif

int main(int argc, char *argv[])
{
#ifdef Q_OS_IOS
    // iOS使用原生UIApplicationMain，完全绕过Qt的应用管理
    @autoreleasepool {
        return UIApplicationMain(argc, argv, nil, NSStringFromClass([AppDelegate class]));
    }
    
#else
    // 非iOS平台的Qt应用程序
    QApplication a(argc, argv);
    
    // 设置Qt日志输出格式，不弹出单独的控制台窗口
    qSetMessagePattern("[%{time yyyyMMdd h:mm:ss.zzz t}] %{type}: %{message}");
    
    qDebug() << "=== GGTPassword Application Starting ===";
    qDebug() << "Debug output enabled";
    
    // 设置应用程序信息
    a.setApplicationName("GGTPassword");
    a.setApplicationVersion("1.0");
    a.setOrganizationName("GGT");
    
    // 解析命令行参数
    QCommandLineParser parser;
    parser.setApplicationDescription("GGTPassword - WebView Bridge Demo");
    parser.addHelpOption();
    parser.addVersionOption();
    
    // 添加调试模式选项
    QCommandLineOption debugOption(QStringList() << "d" << "debug", "Enable debug mode");
    parser.addOption(debugOption);
    
    parser.process(a);
    
    // 确定调试模式
    bool debugMode = false;
    if (parser.isSet(debugOption)) {
        // 使用命令行参数
        debugMode = true;
        qDebug() << "Debug mode enabled via command-line argument";
    } else {
        // 使用编译时设置
#if WEBVIEW_DEBUG_MODE
        debugMode = true;
        qDebug() << "Debug mode enabled via compile-time setting";
#else
        debugMode = false;
        qDebug() << "Debug mode disabled via compile-time setting";
#endif
    }
    
    qDebug() << "Starting WebView Bridge Demo...";
    qDebug() << "Debug mode:" << (debugMode ? "ENABLED" : "DISABLED");

#if WEBVIEW_DEBUG_MODE
    // 初始化DebugManager
    DebugManager::instance()->initialize(debugMode);
#ifdef Q_OS_WIN
    // Windows 下在调试模式时确保标准输出连接到控制台
    if (DebugManager::instance()->isDebugEnabled()) {
        // 尝试附加到父进程控制台（从终端启动时）
        if (!AttachConsole(ATTACH_PARENT_PROCESS)) {
            // 若失败则分配一个新的控制台窗口
            AllocConsole();
        }
        // 将 stdout/stderr 重定向到控制台
        freopen("CONOUT$", "w", stdout);
        freopen("CONOUT$", "w", stderr);
        // 使用 UTF-8 控制台编码
        SetConsoleOutputCP(CP_UTF8);
        SetConsoleCP(CP_UTF8);
        // 安装 Qt 消息处理器，将日志写入 stderr
        qInstallMessageHandler([](QtMsgType type, const QMessageLogContext &ctx, const QString &msg) {
            Q_UNUSED(ctx);
            const char* level = "INFO";
            switch (type) {
                case QtDebugMsg: level = "DEBUG"; break;
                case QtInfoMsg: level = "INFO"; break;
                case QtWarningMsg: level = "WARN"; break;
                case QtCriticalMsg: level = "CRIT"; break;
                case QtFatalMsg: level = "FATAL"; break;
            }
            QByteArray utf8 = msg.toUtf8();
            fprintf(stderr, "[%s] %s\n", level, utf8.constData());
            fflush(stderr);
            if (type == QtFatalMsg) {
                abort();
            }
        });
        qDebug() << "Windows console output attached for debug mode";
    }
#endif
#endif
    
    // 创建MVC组件
    DemoModel* model = new DemoModel();
    DemoController* controller = new DemoController();
    
    // 设置控制器的模型
    controller->setModel(model);
    
    // 创建平台特定的视图
#ifdef Q_OS_MACOS
    MacOSView* view = new MacOSView();
    qDebug() << "Using macOS WebView";
#elif defined(Q_OS_WIN)
    qDebug() << "Creating Windows WebView2...";
    WindowsView* view = new WindowsView();
    qDebug() << "WindowsView created successfully";
    qDebug() << "Using Windows WebView2";
#elif defined(Q_OS_ANDROID)
    qDebug() << "Creating Android WebView...";
    AndroidView* view = new AndroidView();
    qDebug() << "AndroidView created successfully";
    qDebug() << "Using Android WebView";
#else
    // 其他平台暂时使用传统的MainWindow
    MainWindow* w = new MainWindow();
    w->show();
    qDebug() << "Using traditional MainWindow (platform not supported yet)";
    return a.exec();
#endif
    
    qDebug() << "Setting up MVC connections...";
    // 设置控制器和视图的关联（先关联视图，再初始化视图以创建 WebViewBridge）
    view->setController(controller);
    qDebug() << "Initializing view...";
    view->initializeView();
    qDebug() << "View initialized";

    // 视图初始化后绑定 WebViewBridge 到控制器，确保消息处理器正确注册
    qDebug() << "Binding WebViewBridge to controller...";
    controller->setWebViewBridge(view->getWebViewBridge());
    if (!view->getWebViewBridge()) {
        qWarning() << "WebViewBridge is null after view initialization";
    }
    qDebug() << "MVC connections established";

    // 初始化控制器（此时已具备 WebViewBridge，可连接 webViewReady 等信号）
    qDebug() << "Initializing controller...";
    controller->initialize();
    qDebug() << "Controller initialized";
    
    // 加载HTML页面
    qDebug() << "Loading HTML content...";
    QFile htmlFile(":/web/web/index.html");
    if (htmlFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
        QString htmlContent = QString::fromUtf8(htmlFile.readAll());
        view->loadHtml(htmlContent);
        qDebug() << "HTML content loaded successfully from resources";
        qDebug() << "HTML content length:" << htmlContent.length();
    } else {
        qWarning() << "Failed to load HTML file from resources";
        // 备用方案：加载本地文件
        QFile localFile("web/index.html");
        if (localFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
            QString htmlContent = QString::fromUtf8(localFile.readAll());
            view->loadHtml(htmlContent);
            qDebug() << "HTML content loaded from local file";
            qDebug() << "HTML content length:" << htmlContent.length();
        } else {
            qWarning() << "Failed to load HTML file from local path";
            qWarning() << "No HTML content available";
            // 即使没有HTML内容也继续运行，显示空白窗口
        }
    }
    
    // 显示视图
    qDebug() << "Showing view...";
    view->showView();
    qDebug() << "View show() called";
    
    qDebug() << "WebView Bridge Demo started";
    qDebug() << "Window visible:" << view->isVisible();
    qDebug() << "Window geometry:" << view->geometry();
    qDebug() << "Window size:" << view->size();
    qDebug() << "Window position:" << view->pos();
    
    qDebug() << "Entering Qt event loop...";
    int result = a.exec();
    qDebug() << "Qt event loop exited with code:" << result;
    
    // 清理资源
    delete view;
    delete controller;
    delete model;
    
    return result;
#endif
}
