# 架构 - WebView 方案（macOS / iOS / Windows / Android）

| 平台      | 原生视图类型             | 嵌入方式              | 快捷键/输入支持  | 调试支持              | 实现复杂度 | 推荐度   |
| ------- | ------------------ | ----------------- | --------- | ----------------- | ----- | ----- |
| macOS   | NSView (WKWebView) | ✅ 直接 addSubview   | ✅ 完整      | ✅ Safari          | ⭐⭐    | ⭐⭐⭐⭐⭐ |
| iOS     | UIView (WKWebView) | ✅ 直接 addSubview   | ✅ 完整      | ✅ Safari          | ⭐⭐    | ⭐⭐⭐⭐  |
| Windows | HWND (WebView2)    | ⚙️ createWindowEx | ✅ 完整      | ✅ Edge DevTools   | ⭐⭐⭐   | ⭐⭐⭐   |
| Android | Java WebView       | ⚙️ JNI 调用         | ✅ 完整但焦点不稳 | ✅ Chrome DevTools | ⭐⭐⭐⭐  | ⭐⭐    |

## 总体目标
- 使用系统原生 `WKWebView` 承载前端 UI；Qt 只承担逻辑层（`QtCore`），去除 `QtQuick/QtQml` 依赖。
- 统一 JS⇄Native 消息格式为 `{ method, params }`，通过 `window.NativeBridge` 与 `WKScriptMessageHandler("bridge")` 传递。
- 开启 Safari/WebKit 远程调试，定位页面与桥接问题。

---

## macOS：启动与嵌入流程
- 入口视图：`platform/macos/MacOSView.cpp`
  - 构造函数中创建 `MacOSWebViewBridge`，并将其 `QWidget*` 放入 `QVBoxLayout`。
  - `initializeView()` 调用桥接的 `initializeWebView()`。
- 创建与配置 WebView：`platform/macos/MacOSWebViewBridge.mm::setupWebView()`
  - `WKWebViewConfiguration` + `WKUserContentController`。
  - `[userContentController addScriptMessageHandler:m_delegate name:@"bridge"]` 注册原生脚本消息通道。
  - 在 `WEBVIEW_DEBUG_MODE` 下根据 `DebugManager::shouldEnableSafariDevTools()` 打开调试特性：`developerExtrasEnabled`、`setInspectable:YES`（macOS 13.3+）。
  - 创建 `WKWebView` 并设定 `navigationDelegate`。
  - 嵌入到 Qt 控件：`NSView* parent = (__bridge NSView*)m_containerWidget->winId(); [parent addSubview:m_webView];`，并用 AutoLayout 约束贴满父视图。
- 桥接脚本注入：`MacOSWebViewBridge.mm::injectBridgeScript()`
  - 在页面加载后注入：
    ```js
    window.NativeBridge = {
      sendMessage(method, params) {
        window.webkit.messageHandlers.bridge.postMessage(JSON.stringify({method, params: params||{}}));
      },
      onMessage(cb) { window.nativeMessageCallback = cb; }
    };
    window.NativeBridge.sendMessage('bridgeReady', {});
    ```
  - `didFinishNavigation` 中再次注入并触发 `webViewReady` 信号，驱动 `BaseView::viewReady`。
- 加载内容：`MacOSView::loadHtml/loadUrl` 直接调用桥接层向 WKWebView 加载。

## macOS：通信桥接
- JS → Native：`window.NativeBridge.sendMessage('method', { ... })` → `WKScriptMessageHandler('bridge')` → 委托将 JSON 交给 `bridge->handleNativeMessage(qMessage)`。
- Native → JS（建议）：
  - 使用 `evaluateJavaScript(@"window.nativeMessageCallback && window.nativeMessageCallback({method:'nativePong', params:{}})")`。
  - 约定所有下行消息也遵循 `{ method, params }` 格式。

---

## iOS：启动与嵌入流程（推荐统一为 AppDelegate 路径）
- 入口：`UIApplicationMain(argc, argv, nil, NSStringFromClass([AppDelegate class]))`。
- WebView 创建与注入：`platform/ios/AppDelegate.mm::didFinishLaunchingWithOptions`
  - 创建 `UIWindow` 并设为 `rootViewController = IOSViewController`。
  - `WKWebViewConfiguration` + `WKUserContentController`；注册 `bridge`：`[userContentController addScriptMessageHandler:self name:@"bridge"]`。
  - 通过 `WKUserScript` 在 `documentStart` 注入 `window.NativeBridge`（同上），并立即发出 `bridgeReady` 握手。
  - iOS 16.4+ 开启 `webView.inspectable = YES` 以支持远程调试。
  - 创建 `WKWebView` 交给 `IOSViewController::setupWebView` 添加到 `self.view` 并设置约束贴满。
  - 加载 `index.html`（从 App Bundle `web/` 目录，或降级为内置默认 HTML）。
- 说明：项目内还存在 `platform/ios/IOSWebViewBridge.mm`（含自建 WKWebView 与注入逻辑）。为避免“双 WebView/双桥接”冲突，推荐只保留 AppDelegate 这一路径为唯一 WebView 所有者。

## iOS：通信桥接
- JS → Native：同 macOS，通过 `window.webkit.messageHandlers.bridge.postMessage(JSON.stringify(...))`。
- Native → JS：在 `AppDelegate` 内使用：
  ```objc
  [self.webView evaluateJavaScript:@"window.nativeMessageCallback && window.nativeMessageCallback({method:'nativePong', params:{}})"]; 
  ```
- 握手与验证：App 启动后 Xcode 控制台将收到 `bridgeReady`；JS 调用 `sendMessage('ping')` 应在原生日志中可见。

---

## Windows：启动与嵌入流程（WebView2）
- 入口视图：`platform/windows/WindowsView.cpp`
  - 构造 `WindowsWebViewBridge`，挂到 `BaseView` 并连接 `webViewReady → viewReady`。
  - `initializeView()` → `WindowsWebViewBridge::initializeWebView()`。
- 创建与配置 WebView2：`platform/windows/WindowsWebViewBridge.cpp`
  - `initializeCOM()` → `createWebViewEnvironment()` → `createWebViewController()`。
  - 控制器创建后：`resizeWebView()` 设置 `Bounds` 为父 `QWidget` 的大小，随后调用 `setupEventHandlers()`。
  - 事件绑定：
    - `NavigationCompleted`：完成后注入桥接脚本 `injectBridgeScript()` 并触发 `webViewReady`。
    - `WebMessageReceived`：接收 JS → Native 消息，转交 `handleWebMessage(message)` → `WebViewBridge::handleMessage()`。
- 桥接脚本注入：`WindowsWebViewBridge::injectBridgeScript()`
  ```js
  window.NativeBridge = {
    sendMessage(method, params) {
      window.chrome.webview.postMessage(JSON.stringify({method, params: params||{}}));
    },
    onMessage(cb) { window._nativeBridgeCallback = cb; }
  };
  window.handleNativeMessage = function (msg) { window._nativeBridgeCallback && window._nativeBridgeCallback(msg); };
  setTimeout(() => window.NativeBridge.sendMessage('bridgeReady', {}), 100);
  ```
- 加载内容：`WindowsWebViewBridge::loadUrl/loadHtml` 分别使用 `Navigate` 与 `NavigateToString`；首次未初始化时会进入 `pending` 队列，在初始化完成后加载。

## Windows：通信桥接
- JS → Native：`window.chrome.webview.postMessage(JSON.stringify({method, params}))` → WebView2 `WebMessageReceived` → `handleWebMessage()` → 基类解析并发出 `messageReceived`。
- Native → JS：`evaluateJavaScript("window.handleNativeMessage({method:'nativePong', params:{}})")` 下行到页面，页面通过 `onMessage(cb)` 约定接收。

---

## Android：启动与嵌入流程（Java WebView + JNI）
- 入口视图：`platform/android/AndroidView.cpp`
  - 构造 `AndroidWebViewBridge`，连接 `webViewReady` 并调用 `initializeView()`。
  - `setupWebView()` 在 Android UI 线程创建并添加原生 `WebView` 到 `Activity` 的 `contentView`（失败时回退到 `DecorView`），设为全屏、置顶、启用硬件加速与可见性。
- WebView 配置与接口：`platform/android/AndroidWebViewBridge.cpp`
  - `setupWebViewSettings()` 启用 JS、DOM 存储、文件访问、`setAllowUniversalAccessFromFileURLs(true)` 等。
  - `addJavascriptInterface(QtBridgeInterface, "AndroidInterface")` 暴露 Java 方法 `postMessage(String)` 给 JS 调用。
  - `injectBridgeScript()` 在页面注入 `window.QtBridge` 并别名 `window.NativeBridge`：
    ```js
    const QtBridge = {
      sendMessage(method, params) {
        const msg = { method, params: params||{} };
        AndroidInterface && AndroidInterface.postMessage(JSON.stringify(msg));
      },
      onMessage(cb) { window.nativeMessageCallback = cb; },
      handleMessage(message) { /* JSON.parse 后调 nativeMessageCallback */ }
    };
    window.QtBridge = QtBridge; window.NativeBridge = QtBridge;
    ```
- 页面加载完成钩子：`android/src/com/ggt/ggtpassword/LoggingWebViewClient.java::onPageFinished`
  - 再次注入桥接脚本（兜底）。
  - 触发一次 Native→JS 验证：`QtBridge.handleMessage({method:'bridgeReady', params:{url: location.href}})`。
  - 触发一次 JS→Native 验证：`QtBridge.sendMessage('jsPing', { source:'onPageFinished', ts:Date.now() })`。
- 初始化完成：`AndroidWebViewBridge::initializeWebView()` 设置 `m_isInitialized=true` 并发出 `webViewReady`，随后处理 `pendingUrl/pendingHtml` 加载。

## Android：通信桥接
- JS → Native：`AndroidInterface.postMessage(JSON.stringify({method, params}))` → `QtBridgeInterface.java` → JNI → `AndroidWebViewBridge::handleWebViewMessage()` → 基类解析并发出 `messageReceived`。
- Native → JS：`evaluateJavascript("if(window.QtBridge&&window.QtBridge.handleMessage){window.QtBridge.handleMessage({...})}")` 下行到页面，由注入的 `handleMessage` 转交给页面 `onMessage(cb)`。

---

## Web 调试与定位
- macOS：
  - Safari 设置中启用“在菜单栏显示‘开发’菜单”。
  - Debug 模式下 `setInspectable:YES`（macOS 13.3+）与 `developerExtrasEnabled` 已开启。
  - Safari 菜单：`开发 > 本 Mac > GGTPassword > index.html` 打开 Web Inspector。
- iOS：
  - 设备端：`设置 > Safari > 高级 > Web 检查器` 打开；必要时开启开发者模式。
  - Mac 端 Safari 同样打开“开发”菜单；线连设备，Xcode Debug 启动后在 `开发 > [设备名] > GGTPassword > index.html` 连接。
- 快速自检：
  - 控制台执行 `!!window.NativeBridge` 应为 `true`。
  - `window.NativeBridge.sendMessage('ping', { ts: Date.now() })`；Xcode 控制台应看到 “Received message from WebView: …”。
  - Windows：`window.chrome.webview` 存在且 `postMessage` 可调用；Edge DevTools 可连接并看到消息；`handleNativeMessage({...})` 能下发页面。
  - Android：`AndroidInterface` 存在且 `postMessage` 可调用；`chrome://inspect` 能连接；页面能收到 `QtBridge.handleMessage({method:'bridgeReady'})`。

---

## 常见问题与排查
- iOS 崩溃 “There can only be one UIApplication instance”：
  - 原因：同时创建了 `QApplication/QGuiApplication` 与 `UIApplicationMain`，导致重复 UI App 实例。
  - 处理：iOS 路径不要创建 `QApplication/QGuiApplication`；如需 Qt 事件循环，用 `QCoreApplication` 驱动逻辑，并在 `AppDelegate` 完成后初始化。
- 双 WebView/双桥接冲突：
  - 只保留一个 `WKWebView` 所有者（推荐 AppDelegate）。避免同时启用 `IOSWebViewBridge.mm` 的 WKWebView。
- 资源未加载：
  - 确认 App Bundle `web/index.html` 已打包；`pathForResource:@"index" ofType:@"html" inDirectory:@"web"` 返回非空。
- JS 客户端：
  - `index.html` 已内置 `WebViewBridgeClient` 逻辑，会等待 `window.NativeBridge`。如改为外链 `bridge.js`，确保脚本标签正确引入。
- Windows COM 初始化：
  - `CoInitializeEx` 返回 `RPC_E_CHANGED_MODE` 属正常兼容路径；桥接已处理。若 WebView2 环境创建失败，检查 Edge WebView2 运行时是否安装。
- Android 焦点与层级：
  - 若页面无法交互，检查是否正确 `bringToFront()/setZ()/setLayerType(HARDWARE)`；必要时确保添加到 `contentView` 而非 `DecorView`。
- Android 接口名冲突：
  - `addJavascriptInterface(AndroidInterface)` 名字变更需同步修改注入脚本与页面使用；否则 JS→Native 通道不可用。

---

## 关键代码参考
- macOS：
  - `platform/macos/MacOSView.cpp`（承载、布局、信号）
  - `platform/macos/MacOSWebViewBridge.mm`（配置、注入、消息处理、嵌入）
- iOS：
  - `platform/ios/AppDelegate.mm`（窗口、配置、脚本注入、调试、加载）
  - `platform/ios/IOSViewController.mm/.h`（承载与约束）
  - `platform/ios/IOSWebViewBridge.mm`（另一套桥接实现，注意避免并存）
- Windows：
  - `platform/windows/WindowsView.cpp/.h`（承载、初始化、尺寸同步）
  - `platform/windows/WindowsWebViewBridge.cpp/.h`（COM/环境/控制器、事件、脚本注入、消息处理）
- Android：
  - `platform/android/AndroidView.cpp`（承载、初始化、权限与 UI 设置）
  - `platform/android/AndroidWebViewBridge.cpp/.h`（UI 线程创建、设置、接口、脚本注入、JNI 回调）
  - `android/src/com/ggt/ggtpassword/QtBridgeInterface.java`（JSInterface：`postMessage(String)` 通道）
  - `android/src/com/ggt/ggtpassword/LoggingWebViewClient.java`（页面生命周期与桥接脚本兜底注入）

---

> 让 WKWebView 成为系统级子视图，直接参与 Cocoa 事件链：

```objc
// 获取 Qt 窗口对应的 NSView*
void* nsView = reinterpret_cast<void*>(centralWidget->winId());

// 桥接层 Objective-C++
NSView* parent = (__bridge NSView*)nsView;
WKWebView* webView = [[WKWebView alloc] initWithFrame:[parent bounds]];
[parent addSubview:webView];
```