# 键盘快捷键实现方案

## 概述

本文档详细记录了在 macOS 平台上修复 WKWebView 中 Cmd+X/C/V/A/Z/Shift+Z 键盘快捷键不可用问题的完整实现方案。

## 问题背景

在 Qt + WKWebView 的混合应用中，标准的键盘快捷键（如 Cmd+C、Cmd+V 等）无法正常工作，主要原因包括：

1. **事件传递链断裂**：键盘事件在从 macOS 系统传递到 WKWebView 的过程中丢失
2. **焦点管理问题**：WKWebView 无法正确获得和保持键盘焦点
3. **原生菜单缺失**：缺少 macOS 原生菜单系统来处理标准快捷键
4. **Web 页面事件监听缺失**：Web 页面没有监听和处理键盘事件

## 解决方案架构

### 事件传递流程

```
macOS 系统键盘事件
    ↓
Qt 应用程序 (MacOSView)
    ↓
事件过滤器捕获
    ↓
MacOSMenu 处理
    ↓
MacOSWebViewBridge 转发
    ↓
WKWebView JavaScript 执行
    ↓
Web 页面元素操作
```

## 具体实现

### 1. MacOSMenu 类 - 原生菜单系统

**文件位置**: `platform/macos/MacOSMenu.h` 和 `platform/macos/MacOSMenu.mm`

**核心功能**:
- 创建完整的 macOS 原生菜单栏
- 定义标准快捷键（Cmd+Z/Shift+Z/X/C/V/A）
- 将快捷键事件转发给 WebView

**关键代码**:
```cpp
// 创建编辑菜单
void MacOSMenu::createEditMenu()
{
    QMenu* editMenu = m_menuBar->addMenu("编辑");
    
    m_undoAction = editMenu->addAction("撤销");
    m_undoAction->setShortcut(QKeySequence::Undo);  // Cmd+Z
    connect(m_undoAction, &QAction::triggered, this, &MacOSMenu::onUndo);
    
    m_redoAction = editMenu->addAction("重做");
    m_redoAction->setShortcut(QKeySequence::Redo);  // Cmd+Shift+Z
    connect(m_redoAction, &QAction::triggered, this, &MacOSMenu::onRedo);
    
    // ... 其他快捷键
}

// 转发键盘事件到 WebView
void MacOSMenu::forwardKeyEventToWebView(int key, Qt::KeyboardModifiers modifiers)
{
    if (!m_webViewBridge) {
        return;
    }
    
    // 创建 QKeyEvent 并转发
    QKeyEvent keyDownEvent(QEvent::KeyPress, key, modifiers);
    QKeyEvent keyUpEvent(QEvent::KeyRelease, key, modifiers);
    
    m_webViewBridge->sendKeyboardEvent(&keyDownEvent);
    
    QTimer::singleShot(50, [this, key, modifiers]() {
        QKeyEvent keyUpEvent(QEvent::KeyRelease, key, modifiers);
        m_webViewBridge->sendKeyboardEvent(&keyUpEvent);
    });
}
```

### 2. MacOSView 类 - 事件捕获和处理

**文件位置**: `platform/macos/MacOSView.h` 和 `platform/macos/MacOSView.cpp`

**核心功能**:
- 安装事件过滤器捕获键盘事件
- 识别快捷键组合
- 调用菜单系统处理快捷键

**关键代码**:
```cpp
// 事件过滤器
bool MacOSView::eventFilter(QObject *obj, QEvent *event)
{
    if (event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = static_cast<QKeyEvent*>(event);
        
        // 检查是否是 Cmd 键组合
        if (keyEvent->modifiers() & Qt::MetaModifier) {
            handleShortcut(keyEvent);
            return true; // 阻止事件继续传播
        }
    }
    
    return BaseView::eventFilter(obj, event);
}

// 处理快捷键
void MacOSView::handleShortcut(QKeyEvent *event)
{
    int key = event->key();
    Qt::KeyboardModifiers modifiers = event->modifiers();
    
    // 将快捷键转发给菜单处理
    QMetaObject::invokeMethod(m_menu, "forwardKeyEventToWebView", 
                             Qt::QueuedConnection,
                             Q_ARG(int, key),
                             Q_ARG(Qt::KeyboardModifiers, modifiers));
}
```

### 3. MacOSWebViewBridge 类 - 事件转发

**文件位置**: `platform/macos/MacOSWebViewBridge.h` 和 `platform/macos/MacOSWebViewBridge.mm`

**核心功能**:
- 将 Qt 键盘事件转换为 JavaScript 事件
- 管理 WebView 焦点状态
- 通过 JavaScript 执行具体操作

**关键代码**:
```cpp
void MacOSWebViewBridge::sendKeyboardEvent(QKeyEvent* event)
{
    if (!m_webView || !event) {
        return;
    }
    
    // 构造 JavaScript 键盘事件
    QString script = QString(R"(
        (function() {
            var event = new KeyboardEvent('%1', {
                key: '%2',
                code: '%3',
                ctrlKey: %4,
                altKey: %5,
                shiftKey: %6,
                metaKey: %7,
                bubbles: true,
                cancelable: true
            });
            
            var activeElement = document.activeElement;
            if (activeElement) {
                activeElement.dispatchEvent(event);
            } else {
                document.dispatchEvent(event);
            }
        })();
    )")
    .arg(event->type() == QEvent::KeyPress ? "keydown" : "keyup")
    .arg(getKeyString(event->key()))
    .arg(getKeyCode(event->key()))
    .arg(event->modifiers() & Qt::ControlModifier ? "true" : "false")
    .arg(event->modifiers() & Qt::AltModifier ? "true" : "false")
    .arg(event->modifiers() & Qt::ShiftModifier ? "true" : "false")
    .arg(event->modifiers() & Qt::MetaModifier ? "true" : "false");
    
    evaluateJavaScript(script);
}
```

### 4. Web 页面事件监听 - JavaScript 处理

**文件位置**: `web/bridge.js`

**核心功能**:
- 监听键盘事件
- 识别快捷键组合
- 执行具体的编辑操作

**关键代码**:
```javascript
setupKeyboardEventListeners() {
    // 监听键盘事件
    document.addEventListener('keydown', this.handleKeyboardEvent.bind(this), true);
    document.addEventListener('keyup', this.handleKeyboardEvent.bind(this), true);
    
    // 确保页面可以接收键盘事件
    if (document.body) {
        document.body.setAttribute('tabindex', '0');
    }
}

handleKeyboardEvent(event) {
    // 检查是否是我们关心的快捷键组合
    const isCmdOrCtrl = event.metaKey || event.ctrlKey;
    
    if (!isCmdOrCtrl) {
        return; // 不是Cmd/Ctrl组合键，忽略
    }
    
    const key = event.key.toLowerCase();
    const supportedKeys = ['z', 'x', 'c', 'v', 'a'];
    
    if (!supportedKeys.includes(key)) {
        return; // 不是我们支持的键，忽略
    }
    
    // 处理特定的快捷键
    if (event.type === 'keydown') {
        this.handleShortcut(event);
    }
}

performEditAction(action, element, event) {
    try {
        switch (action) {
            case 'undo':
                if (document.execCommand) {
                    document.execCommand('undo');
                }
                break;
            case 'redo':
                if (document.execCommand) {
                    document.execCommand('redo');
                }
                break;
            case 'cut':
                if (document.execCommand) {
                    document.execCommand('cut');
                }
                break;
            case 'copy':
                if (document.execCommand) {
                    document.execCommand('copy');
                }
                break;
            case 'paste':
                this.handlePaste(element);
                break;
            case 'selectAll':
                if (element.select) {
                    element.select();
                } else if (document.execCommand) {
                    document.execCommand('selectAll');
                }
                break;
        }
        
        // 阻止默认行为，避免重复执行
        event.preventDefault();
        event.stopPropagation();
        
    } catch (error) {
        this.log(`执行操作失败: ${action} - ${error.message}`, 'error');
    }
}
```

## 焦点管理

### WKWebView 焦点设置

```objective-c
// 在 setupWebView 中设置焦点
dispatch_async(dispatch_get_main_queue(), ^{
    [[m_webView window] makeFirstResponder:m_webView];
    [m_webView becomeFirstResponder];
});

// 焦点状态检查
bool MacOSWebViewBridge::isWebViewFocused() const
{
    if (!m_webView) {
        return false;
    }
    
    NSWindow* window = [m_webView window];
    if (window) {
        NSResponder* firstResponder = [window firstResponder];
        return (firstResponder == m_webView) || [firstResponder isKindOfClass:[WKWebView class]];
    }
    
    return false;
}
```

### JavaScript 焦点管理

```javascript
setWebViewFocused(focused) {
    if (focused) {
        // 设置WebView为第一响应者
        dispatch_async(dispatch_get_main_queue(), ^{
            NSWindow* window = [m_webView window];
            if (window) {
                [window makeFirstResponder:m_webView];
                [m_webView becomeFirstResponder];
            }
        });
        
        // 通过JavaScript聚焦到可编辑元素
        var editableElements = document.querySelectorAll('input, textarea, [contenteditable="true"]');
        if (editableElements.length > 0) {
            editableElements[0].focus();
        }
    }
}
```

## 支持的快捷键

| 快捷键 | 功能 | 实现方式 |
|--------|------|----------|
| Cmd+Z | 撤销 | document.execCommand('undo') |
| Cmd+Shift+Z | 重做 | document.execCommand('redo') |
| Cmd+X | 剪切 | document.execCommand('cut') |
| Cmd+C | 复制 | document.execCommand('copy') |
| Cmd+V | 粘贴 | Clipboard API + execCommand 回退 |
| Cmd+A | 全选 | element.select() 或 document.execCommand('selectAll') |

## 兼容性处理

### 粘贴操作的现代化处理

```javascript
async handlePaste(element) {
    try {
        // 尝试使用现代的Clipboard API
        if (navigator.clipboard && navigator.clipboard.readText) {
            const text = await navigator.clipboard.readText();
            this.insertTextAtCursor(element, text);
        } else {
            // 回退到execCommand
            if (document.execCommand) {
                document.execCommand('paste');
            }
        }
    } catch (error) {
        this.log(`粘贴操作失败: ${error.message}`, 'error');
    }
}
```

### 跨平台键盘修饰符处理

```javascript
handleShortcut(event) {
    const isMeta = event.metaKey;
    const isCtrl = event.ctrlKey;
    
    // 在macOS上使用Cmd键，其他平台使用Ctrl键
    const isModifierPressed = (this.platform === 'macOS' && isMeta) || 
                             (this.platform !== 'macOS' && isCtrl);
    
    if (!isModifierPressed) {
        return;
    }
    
    // 处理快捷键逻辑...
}
```

## 构建配置

在 `CMakeLists.txt` 中添加新的源文件：

```cmake
if(APPLE AND NOT IOS)
    list(APPEND PROJECT_SOURCES
        # ... 其他文件
        platform/macos/MacOSMenu.h
        platform/macos/MacOSMenu.mm
    )
endif()
```

## 调试和测试

### 日志记录

所有关键步骤都有详细的日志记录：

```javascript
this.log(`键盘事件: ${event.type} - ${event.key}`, 'keyboard', eventInfo);
this.log(`执行快捷键操作: ${action}`, 'shortcut');
this.log(`成功执行操作: ${action}`, 'success');
```

### 错误处理

```javascript
try {
    // 执行操作
} catch (error) {
    this.log(`执行操作失败: ${action} - ${error.message}`, 'error');
}
```

## 总结

通过多层次的事件处理机制，成功实现了在 Qt + WKWebView 混合应用中的键盘快捷键支持：

1. **原生层面**：MacOSMenu 提供标准 macOS 菜单和快捷键
2. **Qt 层面**：MacOSView 捕获和转发键盘事件
3. **桥接层面**：MacOSWebViewBridge 转换事件格式
4. **Web 层面**：JavaScript 执行具体的编辑操作

这种设计确保了键盘快捷键在各种场景下都能正常工作，同时保持了良好的跨平台兼容性和可维护性。