#include "DemoController.h"
#include "DemoModel.h"
#include "../bridge/WebViewBridge.h"
#include <QDebug>
#include <QDateTime>
#include <QSysInfo>

DemoController::DemoController(QObject *parent)
    : BaseController(parent)
{
}

void DemoController::initialize()
{
    qDebug() << "DemoController initialized";
    
    // 连接WebView准备就绪信号
    if (getWebViewBridge()) {
        connect(getWebViewBridge(), &WebViewBridge::webViewReady,
                this, &DemoController::onWebViewReady);
    }
}

void DemoController::handleWebViewMessage(const QString& method, const QJsonObject& params)
{
    qDebug() << "Handling message:" << method << params;
    
    // 根据方法名分发到具体的处理函数
    if (method == "testConnection") {
        handleTestConnection(params);
    } else if (method == "ping") {
        handlePing(params);
    } else if (method == "getSystemInfo") {
        handleGetSystemInfo(params);
    } else if (method == "saveData") {
        handleSaveData(params);
    } else if (method == "loadData") {
        handleLoadData(params);
    } else if (method == "deleteData") {
        handleDeleteData(params);
    } else if (method == "bridgeReady") {
        handleBridgeReady(params);
    } else {
        // 未知方法
        QJsonObject errorData;
        errorData["message"] = QString("Unknown method: %1").arg(method);
        sendResponse(method, errorData, false, "Method not found");
    }
}

void DemoController::onWebViewReady()
{
    qDebug() << "WebView is ready";
    
    // WebView准备就绪后，可以发送初始化数据
    if (getDemoModel()) {
        sendToWebView("initialData", getDemoModel()->toJson());
    }
}

void DemoController::handleTestConnection(const QJsonObject& params)
{
    QJsonObject responseData;
    responseData["message"] = "Connection test successful";
    responseData["timestamp"] = QDateTime::currentDateTime().toString(Qt::ISODate);
    responseData["receivedParams"] = params;
    
    sendResponse("testConnection", responseData);
}

void DemoController::handlePing(const QJsonObject& params)
{
    QJsonObject responseData;
    responseData["message"] = "pong";
    responseData["originalTimestamp"] = params["timestamp"];
    responseData["responseTimestamp"] = QDateTime::currentMSecsSinceEpoch();
    
    sendResponse("ping", responseData);
}

void DemoController::handleGetSystemInfo(const QJsonObject& params)
{
    Q_UNUSED(params)
    
    QJsonObject systemInfo;
    systemInfo["productType"] = QSysInfo::productType();
    systemInfo["productVersion"] = QSysInfo::productVersion();
    systemInfo["kernelType"] = QSysInfo::kernelType();
    systemInfo["kernelVersion"] = QSysInfo::kernelVersion();
    systemInfo["currentCpuArchitecture"] = QSysInfo::currentCpuArchitecture();
    systemInfo["machineHostName"] = QSysInfo::machineHostName();
    
    sendResponse("getSystemInfo", systemInfo);
}

void DemoController::handleSaveData(const QJsonObject& params)
{
    if (getDemoModel() && params.contains("data")) {
        QJsonObject data = params["data"].toObject();
        
        // 模拟保存数据到模型
        if (data.contains("name")) {
            getDemoModel()->setName(data["name"].toString());
        }
        if (data.contains("value")) {
            getDemoModel()->setValue(data["value"].toInt());
        }
        
        getDemoModel()->setStatus("saved");
        
        QJsonObject responseData;
        responseData["message"] = "Data saved successfully";
        responseData["savedData"] = data;
        
        sendResponse("saveData", responseData);
    } else {
        sendResponse("saveData", QJsonObject(), false, "Invalid data format");
    }
}

void DemoController::handleLoadData(const QJsonObject& params)
{
    Q_UNUSED(params)
    
    if (getDemoModel()) {
        QJsonObject responseData;
        responseData["message"] = "Data loaded successfully";
        responseData["data"] = getDemoModel()->toJson();
        
        sendResponse("loadData", responseData);
    } else {
        sendResponse("loadData", QJsonObject(), false, "No model available");
    }
}

void DemoController::handleDeleteData(const QJsonObject& params)
{
    Q_UNUSED(params)
    
    if (getDemoModel()) {
        getDemoModel()->reset();
        getDemoModel()->setStatus("deleted");
        
        QJsonObject responseData;
        responseData["message"] = "Data deleted successfully";
        
        sendResponse("deleteData", responseData);
    } else {
        sendResponse("deleteData", QJsonObject(), false, "No model available");
    }
}

void DemoController::handleBridgeReady(const QJsonObject& params)
{
    Q_UNUSED(params)
    
    qDebug() << "Bridge is ready";
    
    // 发送欢迎消息
    QJsonObject welcomeData;
    welcomeData["message"] = "Welcome to WebView Bridge Demo";
    welcomeData["version"] = "1.0.0";
    welcomeData["platform"] = QSysInfo::productType();
    
    sendToWebView("welcome", welcomeData);
}

DemoModel* DemoController::getDemoModel() const
{
    return qobject_cast<DemoModel*>(getModel());
}