#ifndef DEBUGMANAGER_H
#define DEBUGMANAGER_H

#include <QObject>
#include <QString>

/**
 * @brief DebugManager类 - 统一管理调试功能的开启和关闭
 * 
 * 该类负责：
 * 1. 管理调试模式的全局状态
 * 2. 控制Safari开发者工具的集成
 * 3. 提供调试信息的输出控制
 * 4. 确保非调试模式下完全移除调试功能
 */
class DebugManager : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief 获取DebugManager单例实例
     * @return DebugManager实例指针
     */
    static DebugManager* instance();
    
    /**
     * @brief 初始化调试管理器
     * @param enabled 是否启用调试模式
     */
    void initialize(bool enabled);
    
    /**
     * @brief 检查调试模式是否启用
     * @return true如果调试模式启用，否则false
     */
    bool isDebugEnabled() const;
    
    /**
     * @brief 设置调试模式状态
     * @param enabled 调试模式状态
     */
    void setDebugEnabled(bool enabled);
    
    /**
     * @brief 输出调试信息（仅在调试模式下）
     * @param message 调试消息
     * @param category 消息分类（可选）
     */
    void debugLog(const QString& message, const QString& category = "General");
    
    /**
     * @brief 检查是否应该启用Safari开发者工具
     * @return true如果应该启用Safari开发者工具
     */
    bool shouldEnableSafariDevTools() const;
    
    /**
     * @brief 获取调试模式描述信息
     * @return 调试模式状态描述
     */
    QString getDebugModeDescription() const;

signals:
    /**
     * @brief 调试模式状态改变信号
     * @param enabled 新的调试模式状态
     */
    void debugModeChanged(bool enabled);

private:
    explicit DebugManager(QObject *parent = nullptr);
    ~DebugManager();
    
    // 禁用拷贝构造和赋值操作
    DebugManager(const DebugManager&) = delete;
    DebugManager& operator=(const DebugManager&) = delete;
    
    bool m_debugEnabled;
    static DebugManager* s_instance;
};

// 便利宏定义
#if WEBVIEW_DEBUG_MODE
    #define DEBUG_LOG(message) DebugManager::instance()->debugLog(message)
    #define DEBUG_LOG_CATEGORY(message, category) DebugManager::instance()->debugLog(message, category)
    #define IS_DEBUG_ENABLED() DebugManager::instance()->isDebugEnabled()
#else
    #define DEBUG_LOG(message) do {} while(0)
    #define DEBUG_LOG_CATEGORY(message, category) do {} while(0)
    #define IS_DEBUG_ENABLED() false
#endif

#endif // DEBUGMANAGER_H