#include "DebugManager.h"
#include <QDebug>
#include <QDateTime>

DebugManager* DebugManager::s_instance = nullptr;

DebugManager::DebugManager(QObject *parent)
    : QObject(parent)
    , m_debugEnabled(false)
{
}

DebugManager::~DebugManager()
{
}

DebugManager* DebugManager::instance()
{
    if (!s_instance) {
        s_instance = new DebugManager();
    }
    return s_instance;
}

void DebugManager::initialize(bool enabled)
{
    bool oldState = m_debugEnabled;
    m_debugEnabled = enabled;
    
    if (oldState != enabled) {
        emit debugModeChanged(enabled);
    }
    
#if WEBVIEW_DEBUG_MODE
    if (enabled) {
        qDebug() << "=== DEBUG MODE INITIALIZED ===";
        qDebug() << "Timestamp:" << QDateTime::currentDateTime().toString();
        qDebug() << "Safari Developer Tools: ENABLED";
        qDebug() << "Debug Logging: ENABLED";
        qDebug() << "===============================";
    } else {
        qDebug() << "Debug mode available but disabled by runtime setting";
    }
#else
    Q_UNUSED(enabled)
    // 在非调试编译模式下，不输出任何调试信息
#endif
}

bool DebugManager::isDebugEnabled() const
{
#if WEBVIEW_DEBUG_MODE
    return m_debugEnabled;
#else
    return false;
#endif
}

void DebugManager::setDebugEnabled(bool enabled)
{
#if WEBVIEW_DEBUG_MODE
    if (m_debugEnabled != enabled) {
        m_debugEnabled = enabled;
        emit debugModeChanged(enabled);
        
        debugLog(enabled ? "Debug mode enabled" : "Debug mode disabled", "DebugManager");
    }
#else
    Q_UNUSED(enabled)
    // 在非调试编译模式下，忽略设置请求
#endif
}

void DebugManager::debugLog(const QString& message, const QString& category)
{
#if WEBVIEW_DEBUG_MODE
    if (m_debugEnabled) {
        QString timestamp = QDateTime::currentDateTime().toString("hh:mm:ss.zzz");
        qDebug() << QString("[%1][%2] %3").arg(timestamp, category, message);
    }
#else
    Q_UNUSED(message)
    Q_UNUSED(category)
    // 在非调试编译模式下，不输出任何信息
#endif
}

bool DebugManager::shouldEnableSafariDevTools() const
{
#if WEBVIEW_DEBUG_MODE
    return m_debugEnabled;
#else
    return false;
#endif
}

QString DebugManager::getDebugModeDescription() const
{
#if WEBVIEW_DEBUG_MODE
    if (m_debugEnabled) {
        return "Debug mode: ENABLED (Safari DevTools available, Debug logging active)";
    } else {
        return "Debug mode: DISABLED (Debug features compiled but not active)";
    }
#else
    return "Debug mode: DISABLED (No debug features compiled)";
#endif
}