#!/bin/bash

# iOS构建脚本
# 支持真机和模拟器构建

set -e  # 遇到错误立即退出

# 颜色输出
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 打印带颜色的消息
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 显示帮助信息
show_help() {
    echo "iOS构建脚本"
    echo ""
    echo "用法: $0 [选项]"
    echo ""
    echo "选项:"
    echo "  -t, --target TARGET    构建目标 (device|simulator|both) [默认: both]"
    echo "  -c, --config CONFIG    构建配置 (Debug|Release) [默认: Debug]"
    echo "  -d, --team-id TEAM_ID  开发团队ID"
    echo "  -i, --identity IDENTITY 代码签名身份 [默认: iPhone Developer]"
    echo "  -b, --bundle-id BUNDLE_ID  指定Bundle Identifier (例如 com.yourcorp.app)"
    echo "  -h, --help            显示此帮助信息"
    echo ""
    echo "示例:"
    echo "  $0 -t device -c Release -d ABCD123456"
    echo "  $0 -t simulator"
    echo "  $0 --target both --config Debug"
}

# 默认参数
TARGET="both"
CONFIG="Debug"
TEAM_ID=""
IDENTITY="iPhone Developer"
PROJECT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BUILD_DIR="$PROJECT_DIR/build-ios"
BUNDLE_ID=""

# 解析命令行参数
while [[ $# -gt 0 ]]; do
    case $1 in
        -t|--target)
            TARGET="$2"
            shift 2
            ;;
        -c|--config)
            CONFIG="$2"
            shift 2
            ;;
        -d|--team-id)
            TEAM_ID="$2"
            shift 2
            ;;
        -i|--identity)
            IDENTITY="$2"
            shift 2
            ;;
        -h|--help)
            show_help
            exit 0
            ;;
        -b|--bundle-id)
            BUNDLE_ID="$2"
            shift 2
            ;;
        *)
            print_error "未知参数: $1"
            show_help
            exit 1
            ;;
    esac
done

# 验证参数
if [[ "$TARGET" != "device" && "$TARGET" != "simulator" && "$TARGET" != "both" ]]; then
    print_error "无效的构建目标: $TARGET"
    show_help
    exit 1
fi

if [[ "$CONFIG" != "Debug" && "$CONFIG" != "Release" ]]; then
    print_error "无效的构建配置: $CONFIG"
    show_help
    exit 1
fi

print_info "开始iOS构建..."
print_info "项目目录: $PROJECT_DIR"
print_info "构建目录: $BUILD_DIR"
print_info "构建目标: $TARGET"
print_info "构建配置: $CONFIG"
if [[ -n "$BUNDLE_ID" ]]; then
    print_info "Bundle Identifier: $BUNDLE_ID"
else
    print_info "Bundle Identifier: 使用项目默认配置"
fi

# 检查必要工具
print_info "检查构建环境..."

if ! command -v cmake &> /dev/null; then
    print_error "CMake未安装或不在PATH中"
    exit 1
fi

if ! command -v xcodebuild &> /dev/null; then
    print_error "Xcode未安装或不在PATH中"
    exit 1
fi

# 检查Qt for iOS
print_info "检查Qt for iOS安装..."

# 优先使用官方Qt安装
if [[ -d "$HOME/Qt/6.10.0/ios" ]]; then
    QT_CMAKE_DIR="$HOME/Qt/6.10.0/ios/lib/cmake/Qt6"
    QT_HOST_PATH="$HOME/Qt/6.10.0/macos"
    print_info "使用官方Qt 6.10.0 iOS安装"
elif [[ -d "$HOME/Qt/6.9.0/ios" ]]; then
    QT_CMAKE_DIR="$HOME/Qt/6.9.0/ios/lib/cmake/Qt6"
    QT_HOST_PATH="$HOME/Qt/6.9.0/macos"
    print_info "使用官方Qt 6.9.0 iOS安装"
elif [[ -d "/opt/homebrew/Cellar/qt" ]]; then
    QT_VERSION=$(ls "/opt/homebrew/Cellar/qt" | head -1)
    QT_CMAKE_DIR="/opt/homebrew/Cellar/qt/$QT_VERSION/lib/cmake/Qt6"
    QT_HOST_PATH=""
    print_warning "使用Homebrew Qt安装（可能不支持iOS）"
else
    print_error "未找到Qt for iOS安装"
    print_error "请安装Qt for iOS或使用Qt Maintenance Tool添加iOS组件"
    exit 1
fi

if [[ ! -d "$QT_CMAKE_DIR" ]]; then
    print_error "Qt CMake目录不存在: $QT_CMAKE_DIR"
    exit 1
fi

print_success "构建环境检查完成"
print_info "Qt6 CMake目录: $QT_CMAKE_DIR"

# 构建函数
build_for_platform() {
    local platform=$1
    local sdk=$2
    local arch=$3
    local build_suffix=$4
    
    print_info "开始构建 $platform ($arch)..."
    
    local current_build_dir="$BUILD_DIR-$build_suffix"
    
    # 清理并创建构建目录
    rm -rf "$current_build_dir"
    mkdir -p "$current_build_dir"
    cd "$current_build_dir"
    
    # CMake配置参数
    local cmake_args=(
        -G "Xcode"
        -DCMAKE_SYSTEM_NAME=iOS
        -DCMAKE_OSX_SYSROOT="$sdk"
        -DCMAKE_OSX_ARCHITECTURES="$arch"
        -DCMAKE_BUILD_TYPE="$CONFIG"
        -DCMAKE_PREFIX_PATH="$QT_CMAKE_DIR"
        -DQt6_DIR="$QT_CMAKE_DIR"
    )
    
    # 如果有host path，添加相关参数
    if [ -n "$QT_HOST_PATH" ]; then
        cmake_args+=("-DQT_HOST_PATH=$QT_HOST_PATH")
        cmake_args+=("-DQT_HOST_PATH_CMAKE_DIR=$QT_HOST_PATH/lib/cmake")
    fi
    
    print_info "CMake参数: ${cmake_args[*]}"
    
    # 添加开发团队ID（如果提供）
    if [[ -n "$TEAM_ID" ]]; then
        cmake_args+=(-DCMAKE_XCODE_ATTRIBUTE_DEVELOPMENT_TEAM="$TEAM_ID")
    fi

    # 添加代码签名身份
    cmake_args+=(-DCMAKE_XCODE_ATTRIBUTE_CODE_SIGN_IDENTITY="$IDENTITY")

    # 覆盖Bundle Identifier（如果提供）
    if [[ -n "$BUNDLE_ID" ]]; then
        cmake_args+=(-DBUNDLE_IDENTIFIER="$BUNDLE_ID")
    fi
    
    print_info "运行CMake配置..."
    
    # 临时清除可能干扰的环境变量，避免混用 Homebrew 的 Qt
    export CPATH=""
    export CPLUS_INCLUDE_PATH=""
    export C_INCLUDE_PATH=""
    export CMAKE_PREFIX_PATH="$QT_CMAKE_DIR"
    
    # 使用qt-cmake工具进行配置
    QT_CMAKE_TOOL="$QT_CMAKE_DIR/../../../bin/qt-cmake"
    if [ -f "$QT_CMAKE_TOOL" ]; then
        print_info "使用qt-cmake工具: $QT_CMAKE_TOOL"
        
        # 为模拟器构建添加额外参数以正确引用XCFramework中的模拟器切片
        if [[ "$sdk" == "iphonesimulator" ]]; then
            "$QT_CMAKE_TOOL" -G "Xcode" \
                -DCMAKE_SYSTEM_NAME=iOS \
                -DCMAKE_OSX_SYSROOT="$sdk" \
                -DCMAKE_OSX_ARCHITECTURES="$arch" \
                -DCMAKE_BUILD_TYPE="$CONFIG" \
                -DCMAKE_IOS_INSTALL_COMBINED=YES \
                -DCMAKE_FIND_FRAMEWORK=FIRST \
                -DCMAKE_FIND_APPBUNDLE=FIRST \
                ${BUNDLE_ID:+-DBUNDLE_IDENTIFIER="$BUNDLE_ID"} \
                "$PROJECT_DIR"
        else
            "$QT_CMAKE_TOOL" -G "Xcode" \
                -DCMAKE_SYSTEM_NAME=iOS \
                -DCMAKE_OSX_SYSROOT="$sdk" \
                -DCMAKE_OSX_ARCHITECTURES="$arch" \
                -DCMAKE_BUILD_TYPE="$CONFIG" \
                -DCMAKE_FIND_FRAMEWORK=FIRST \
                -DCMAKE_FIND_APPBUNDLE=FIRST \
                ${BUNDLE_ID:+-DBUNDLE_IDENTIFIER="$BUNDLE_ID"} \
                "$PROJECT_DIR"
        fi
    else
        print_warning "qt-cmake工具未找到，使用传统cmake方式"
        cmake "${cmake_args[@]}" "$PROJECT_DIR"
    fi
    
    if [[ $? -ne 0 ]]; then
        print_error "CMake配置失败"
        return 1
    fi
    
    print_info "开始编译..."
    
    # 根据目标平台选择不同的构建参数
    if [[ "$platform_type" == "device" ]]; then
        # 设备构建：使用xcodebuild直接构建以支持自动签名
        print_info "使用xcodebuild进行设备构建..."
        xcodebuild -project GGTPassword.xcodeproj \
                   -scheme GGTPassword \
                   -configuration "$CONFIG" \
                   -destination "generic/platform=iOS" \
                   -allowProvisioningUpdates \
                   build
        
        if [[ $? -ne 0 ]]; then
            print_warning "自动签名构建失败，尝试使用cmake构建..."
            # 备用方案：使用cmake构建
            cmake --build . --config "$CONFIG"
        fi
    else
        # 模拟器构建：使用 x86_64 架构（Qt 6.10.0 限制）
        print_info "使用xcodebuild构建模拟器（x86_64架构）..."
        xcodebuild -project GGTPassword.xcodeproj \
                   -scheme GGTPassword \
                   -configuration "$CONFIG" \
                   -destination "generic/platform=iOS Simulator" \
                   ARCHS="x86_64" \
                   ONLY_ACTIVE_ARCH=NO \
                   build
    fi
    
    if [[ $? -ne 0 ]]; then
        print_error "编译失败"
        return 1
    fi
    
    print_success "$platform ($arch) 构建完成"
    
    # 显示输出文件位置
    local app_path="$current_build_dir/$CONFIG-$sdk/GGTPassword.app"
    if [[ -d "$app_path" ]]; then
        print_success "应用包位置: $app_path"
    fi
    
    return 0
}

# 执行构建
cd "$PROJECT_DIR"

if [[ "$TARGET" == "device" || "$TARGET" == "both" ]]; then
    build_for_platform "iOS Device" "iphoneos" "arm64" "device"
    if [[ $? -ne 0 ]]; then
        print_error "iOS设备构建失败"
        exit 1
    fi
fi

if [[ "$TARGET" == "simulator" || "$TARGET" == "both" ]]; then
    build_for_platform "iOS Simulator" "iphonesimulator" "x86_64" "simulator"
    if [[ $? -ne 0 ]]; then
        print_error "iOS模拟器构建失败"
        exit 1
    fi
fi

print_success "所有构建任务完成！"

# 显示下一步操作提示
echo ""
print_info "下一步操作:"
if [[ "$TARGET" == "device" || "$TARGET" == "both" ]]; then
    echo "  • 真机部署: 使用Xcode打开 $BUILD_DIR-device/GGTPassword.xcodeproj"
fi
if [[ "$TARGET" == "simulator" || "$TARGET" == "both" ]]; then
    echo "  • 模拟器测试: 使用Xcode打开 $BUILD_DIR-simulator/GGTPassword.xcodeproj"
fi
echo "  • 或者使用xcodebuild命令行工具进行部署"