#!/bin/bash

# Android编译脚本
# 用于编译GGTPassword Android版本

set -e  # 遇到错误时退出

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 打印带颜色的消息
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 检查必要的环境变量和工具
check_environment() {
    print_info "检查Android开发环境..."
    
    # 检查Android SDK（支持默认路径）
    if [ -z "$ANDROID_SDK_ROOT" ] && [ -z "$ANDROID_HOME" ]; then
        DEFAULT_SDK="$HOME/Library/Android/sdk"
        if [ -d "$DEFAULT_SDK" ]; then
            ANDROID_SDK_ROOT="$DEFAULT_SDK"
            export ANDROID_SDK_ROOT
            print_info "使用默认Android SDK路径: $ANDROID_SDK_ROOT"
        else
            print_error "未设置ANDROID_SDK_ROOT或ANDROID_HOME环境变量，且默认路径不存在"
            print_info "请设置Android SDK路径，例如："
            print_info "export ANDROID_SDK_ROOT=\$HOME/Library/Android/sdk"
            exit 1
        fi
    fi
    
    # 使用ANDROID_SDK_ROOT，如果未设置则使用ANDROID_HOME
    if [ -n "$ANDROID_SDK_ROOT" ]; then
        ANDROID_SDK="$ANDROID_SDK_ROOT"
    else
        ANDROID_SDK="$ANDROID_HOME"
    fi
    # 检查路径是否存在
    if [ ! -d "$ANDROID_SDK" ]; then
        print_error "Android SDK路径不存在: $ANDROID_SDK"
        print_info "请设置ANDROID_SDK_ROOT或ANDROID_HOME，例如："
        print_info "export ANDROID_SDK_ROOT=\$HOME/Library/Android/sdk"
        exit 1
    fi
    
    # 检查Android NDK
    if [ -z "$ANDROID_NDK_ROOT" ]; then
        # 尝试从SDK中查找NDK
        if [ -d "$ANDROID_SDK/ndk" ]; then
            # 查找最新版本的NDK
            ANDROID_NDK_ROOT=$(find "$ANDROID_SDK/ndk" -maxdepth 1 -type d | sort -V | tail -1)
            if [ -n "$ANDROID_NDK_ROOT" ]; then
                print_info "找到NDK: $ANDROID_NDK_ROOT"
                export ANDROID_NDK_ROOT
            fi
        fi
        
        if [ -z "$ANDROID_NDK_ROOT" ]; then
            print_error "未找到Android NDK"
            print_info "请安装Android NDK或设置ANDROID_NDK_ROOT环境变量"
            exit 1
        fi
    fi
    
    # 检查Qt for Android（支持默认路径）
    if [ -z "$QT_ANDROID_ROOT" ]; then
        DEFAULT_QT="$HOME/Qt/6.10.0/android_arm64_v8a"
        if [ -d "$DEFAULT_QT" ]; then
            QT_ANDROID_ROOT="$DEFAULT_QT"
            export QT_ANDROID_ROOT
            print_info "使用默认Qt for Android路径: $QT_ANDROID_ROOT"
        else
            print_error "未设置QT_ANDROID_ROOT环境变量，且默认路径不存在"
            print_info "请设置Qt for Android的安装路径，例如："
            print_info "export QT_ANDROID_ROOT=\$HOME/Qt/6.10.0/android_arm64_v8a"
            exit 1
        fi
    fi
    
    if [ ! -d "$QT_ANDROID_ROOT" ]; then
        print_error "Qt for Android路径不存在: $QT_ANDROID_ROOT"
        print_info "请设置QT_ANDROID_ROOT，例如："
        print_info "export QT_ANDROID_ROOT=\$HOME/Qt/6.10.0/android_arm64_v8a"
        exit 1
    fi
    
    # 检查CMake
    if ! command -v cmake &> /dev/null; then
        print_error "未找到CMake"
        print_info "请安装CMake"
        exit 1
    fi
    
    # 检查Ninja（可选）
    if command -v ninja &> /dev/null; then
        CMAKE_GENERATOR="Ninja"
        print_info "使用Ninja构建系统"
    else
        CMAKE_GENERATOR="Unix Makefiles"
        print_info "使用Make构建系统"
    fi
    
    print_success "环境检查完成"
}

# 设置构建参数
setup_build_config() {
    print_info "设置构建配置..."
    
    # 默认参数
    BUILD_TYPE=${BUILD_TYPE:-Release}
    ANDROID_ABI=${ANDROID_ABI:-arm64-v8a}
    # CMake 使用的 NDK 平台（上限 35），默认 33
    ANDROID_PLATFORM=${ANDROID_PLATFORM:-android-33}
    # androiddeployqt/Gradle 使用的 SDK 平台（使用最高已安装）
    ANDROID_DEPLOY_PLATFORM=""
    AVAILABLE_ALL=$(ls -1 "$ANDROID_SDK/platforms" 2>/dev/null | grep -E '^android-[0-9]+' | sed 's/android-//' | sort -n)
    HIGHEST_ALL=$(echo "$AVAILABLE_ALL" | tail -1)
    if [ -n "$HIGHEST_ALL" ] && [ -d "$ANDROID_SDK/platforms/android-$HIGHEST_ALL" ]; then
        ANDROID_DEPLOY_PLATFORM="android-$HIGHEST_ALL"
    else
        print_error "未找到任何已安装的Android平台，请通过 SDK Manager 安装。"
        exit 1
    fi
    # 将 NDK 平台号限制到 35
    NUM=${ANDROID_PLATFORM#android-}
    if [ "$NUM" -gt 35 ]; then
        ANDROID_PLATFORM="android-35"
    fi
    BUILD_DIR=${BUILD_DIR:-build-android-${ANDROID_ABI}}
    
    # 创建构建目录
    if [ -d "$BUILD_DIR" ]; then
        print_warning "构建目录已存在: $BUILD_DIR"
        read -p "是否清理并重新构建? (y/N): " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            rm -rf "$BUILD_DIR"
            print_info "已清理构建目录"
        fi
    fi
    
    mkdir -p "$BUILD_DIR"
    
    print_info "构建配置:"
    print_info "  构建类型: $BUILD_TYPE"
    print_info "  Android ABI: $ANDROID_ABI"
    print_info "  Android平台(NDK/CMake): $ANDROID_PLATFORM"
    print_info "  Android平台(androiddeployqt/Gradle): $ANDROID_DEPLOY_PLATFORM"
    print_info "  构建目录: $BUILD_DIR"
    print_info "  Qt路径: $QT_ANDROID_ROOT"
    print_info "  Android SDK: $ANDROID_SDK"
    print_info "  Android NDK: $ANDROID_NDK_ROOT"
}

# 配置CMake
configure_cmake() {
    print_info "配置CMake..."
    
    cd "$BUILD_DIR"

    # 在配置前生成构建目录下的 Android 包源（避免污染源码）
    PACKAGE_DIR="$PWD/android-package"
    SRC_SRC="../platform/android/src"
    SRC_DST="$PACKAGE_DIR/src"
    MANIFEST_SRC="../platform/android/AndroidManifest.xml"
    MANIFEST_DST="$PACKAGE_DIR/AndroidManifest.xml"
    ASSETS_SRC="../web"
    ASSETS_DST="$PACKAGE_DIR/assets/web"

    print_info "准备打包源目录: $PACKAGE_DIR"
    mkdir -p "$PACKAGE_DIR" "$SRC_DST" "$ASSETS_DST"

    # 同步 Java/Kotlin 源码
    if [ -d "$SRC_SRC" ]; then
        print_info "同步Android src: $SRC_SRC -> $SRC_DST"
        rsync -a --delete "$SRC_SRC/" "$SRC_DST/" || (rm -rf "$SRC_DST" && cp -R "$SRC_SRC" "$SRC_DST")
    else
        print_warning "未找到src目录: $SRC_SRC"
    fi

    # 同步 Manifest
    if [ -f "$MANIFEST_SRC" ]; then
        print_info "复制AndroidManifest: $MANIFEST_SRC -> $MANIFEST_DST"
        cp "$MANIFEST_SRC" "$MANIFEST_DST"
    else
        print_error "缺少AndroidManifest.xml: $MANIFEST_SRC"
        exit 1
    fi

    # 同步 Web 前端到 assets/web
    if [ -d "$ASSETS_SRC" ]; then
        print_info "同步Android assets: $ASSETS_SRC -> $ASSETS_DST"
        rsync -a --delete "$ASSETS_SRC/" "$ASSETS_DST/" || (rm -rf "$ASSETS_DST" && cp -R "$ASSETS_SRC" "$ASSETS_DST")
    else
        print_warning "未找到web目录: $ASSETS_SRC，若使用 WebView 本地页面请确认资源路径"
    fi
    
    # CMake配置参数
    CMAKE_ARGS=(
        -DCMAKE_BUILD_TYPE="$BUILD_TYPE"
        -DCMAKE_TOOLCHAIN_FILE="$ANDROID_NDK_ROOT/build/cmake/android.toolchain.cmake"
        -DANDROID_ABI="$ANDROID_ABI"
        -DANDROID_PLATFORM="$ANDROID_PLATFORM"
        -DANDROID_STL=c++_shared
        -DCMAKE_PREFIX_PATH="$QT_ANDROID_ROOT"
        -DQt6_DIR="$QT_ANDROID_ROOT/lib/cmake/Qt6"
        -DQt6Core_DIR="$QT_ANDROID_ROOT/lib/cmake/Qt6Core"
        -DQt6Gui_DIR="$QT_ANDROID_ROOT/lib/cmake/Qt6Gui"
        -DQt6Widgets_DIR="$QT_ANDROID_ROOT/lib/cmake/Qt6Widgets"
        -DQt6WebView_DIR="$QT_ANDROID_ROOT/lib/cmake/Qt6WebView"
        -DQT_ANDROID_ROOT="$QT_ANDROID_ROOT"
        -DANDROID_SDK_ROOT="$ANDROID_SDK"
        -DANDROID_NDK_ROOT="$ANDROID_NDK_ROOT"
        -DGGT_ANDROID_PACKAGE_SOURCE_DIR="$PACKAGE_DIR"
        -G "$CMAKE_GENERATOR"
    )
    
    # 添加调试选项
    if [ "$BUILD_TYPE" = "Debug" ]; then
        CMAKE_ARGS+=(-DDEBUG_MODE=ON)
    else
        CMAKE_ARGS+=(-DDEBUG_MODE=OFF)
    fi
    
    print_info "执行CMake配置..."
    cmake "${CMAKE_ARGS[@]}" ..
    
    if [ $? -eq 0 ]; then
        print_success "CMake配置完成"
    else
        print_error "CMake配置失败"
        exit 1
    fi
    
    cd ..
}

# 编译项目
build_project() {
    print_info "开始编译..."
    
    cd "$BUILD_DIR"
    
    # 获取CPU核心数用于并行编译
    if command -v nproc &> /dev/null; then
        JOBS=$(nproc)
    elif [ -f /proc/cpuinfo ]; then
        JOBS=$(grep -c ^processor /proc/cpuinfo)
    else
        JOBS=4
    fi
    
    print_info "使用 $JOBS 个并行任务进行编译"
    
    if [ "$CMAKE_GENERATOR" = "Ninja" ]; then
        ninja -j "$JOBS"
    else
        make -j "$JOBS"
    fi
    
    if [ $? -eq 0 ]; then
        print_success "编译完成"
    else
        print_error "编译失败"
        exit 1
    fi
    
    cd ..
}

# 生成APK
generate_apk() {
    print_info "生成APK..."
    
    cd "$BUILD_DIR"
    
    # 查找生成的库文件
    LIB_FILE=$(find . -name "libGGTPassword*.so" | head -1)
    
    if [ -z "$LIB_FILE" ]; then
        print_error "未找到编译生成的库文件"
        exit 1
    fi
    
    print_info "找到库文件: $LIB_FILE"
    
    # 使用Qt的androiddeployqt工具生成APK
    # androiddeployqt工具在Qt的macos目录中
    QT_BASE_DIR=$(dirname "$QT_ANDROID_ROOT")
    ANDROIDDEPLOYQT="$QT_BASE_DIR/macos/bin/androiddeployqt"
    
    if [ ! -f "$ANDROIDDEPLOYQT" ]; then
        print_error "未找到androiddeployqt工具: $ANDROIDDEPLOYQT"
        exit 1
    fi
    
    # 统一使用 platform/android 作为打包目录；如缺失则创建基本模板
    PACKAGE_DIR="../platform/android"
    if [ ! -d "$PACKAGE_DIR" ]; then
        print_info "创建Android打包目录: $PACKAGE_DIR"
        mkdir -p "$PACKAGE_DIR"
    fi

    if [ ! -f "$PACKAGE_DIR/AndroidManifest.xml" ]; then
        print_info "创建基本的AndroidManifest.xml 至 $PACKAGE_DIR"
        cat > "$PACKAGE_DIR/AndroidManifest.xml" << EOF
<?xml version="1.0"?>
<manifest package="com.ggt.ggtpassword" 
          xmlns:android="http://schemas.android.com/apk/res/android" 
          android:versionName="1.0" 
          android:versionCode="1" 
          android:installLocation="auto">
    
    <uses-permission android:name="android.permission.INTERNET" />
    <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE" />
    
    <application android:hardwareAccelerated="true" 
                 android:name="org.qtproject.qt.android.bindings.QtApplication" 
                 android:label="GGTPassword"
                 android:extractNativeLibs="true"
                 android:theme="@android:style/Theme.NoTitleBar.Fullscreen">
        <meta-data android:name="android.app.lib_name" android:value="GGTPassword"/>
        
        <activity android:configChanges="orientation|uiMode|screenLayout|screenSize|smallestScreenSize|layoutDirection|locale|fontScale|keyboard|keyboardHidden|navigation|mcc|mnc|density"
                  android:name="org.qtproject.qt5.android.bindings.QtActivity"
                  android:label="GGTPassword"
                  android:screenOrientation="unspecified"
                  android:launchMode="singleTop"
                  android:exported="true">
            <intent-filter>
                <action android:name="android.intent.action.MAIN"/>
                <category android:name="android.intent.category.LAUNCHER"/>
            </intent-filter>
        </activity>
    </application>
    
    <uses-sdk android:minSdkVersion="24" android:targetSdkVersion="33"/>
</manifest>
EOF
    fi
    
    # CMake 已构建 Gradle 项目并生成 APK，直接查找输出
    APK_FILE=$(find ./android-build/build/outputs/apk -name "*.apk" | head -1)
    if [ -n "$APK_FILE" ]; then
        print_success "APK文件: $APK_FILE"
    else
        print_warning "未在 android-build 输出目录找到 APK，跳过二次打包。"
    fi

    cd ..
}

# 清理函数
clean_build() {
    print_info "清理构建文件..."
    
    if [ -d "$BUILD_DIR" ]; then
        rm -rf "$BUILD_DIR"
        print_success "已清理构建目录: $BUILD_DIR"
    fi
    
    if [ -d "android" ]; then
        rm -rf "android"
        print_success "已清理Android输出目录"
    fi
}

# 显示帮助信息
show_help() {
    echo "Android编译脚本 - GGTPassword"
    echo ""
    echo "用法: $0 [选项]"
    echo ""
    echo "选项:"
    echo "  -h, --help              显示此帮助信息"
    echo "  -c, --clean             清理构建文件"
    echo "  -t, --type TYPE         构建类型 (Debug|Release，默认: Release)"
    echo "  -a, --abi ABI           Android ABI (arm64-v8a|armeabi-v7a|x86|x86_64，默认: arm64-v8a)"
    echo "  -p, --platform PLATFORM Android平台版本 (默认: android-24)"
    echo "  -d, --dir DIR           构建目录 (默认: build-android-\${ABI})"
    echo "  --no-apk                只编译，不生成APK"
    echo ""
    echo "环境变量:"
    echo "  ANDROID_SDK_ROOT        Android SDK路径（默认: \$HOME/Library/Android/sdk）"
    echo "  ANDROID_NDK_ROOT        Android NDK路径"
    echo "  QT_ANDROID_ROOT         Qt for Android路径（默认: \$HOME/Qt/6.10.0/android_arm64_v8a）"
    echo "  JAVA_HOME               Java JDK路径"
    echo ""
    echo "示例:"
    echo "  $0                      # 使用默认配置编译"
    echo "  $0 -t Debug -a armeabi-v7a  # 编译Debug版本，ARM 32位"
    echo "  $0 --clean              # 清理构建文件"
}

# 主函数
main() {
    local CLEAN_ONLY=false
    local GENERATE_APK=true
    
    # 解析命令行参数
    while [[ $# -gt 0 ]]; do
        case $1 in
            -h|--help)
                show_help
                exit 0
                ;;
            -c|--clean)
                CLEAN_ONLY=true
                shift
                ;;
            -t|--type)
                BUILD_TYPE="$2"
                shift 2
                ;;
            -a|--abi)
                ANDROID_ABI="$2"
                shift 2
                ;;
            -p|--platform)
                ANDROID_PLATFORM="$2"
                shift 2
                ;;
            -d|--dir)
                BUILD_DIR="$2"
                shift 2
                ;;
            --no-apk)
                GENERATE_APK=false
                shift
                ;;
            *)
                print_error "未知选项: $1"
                show_help
                exit 1
                ;;
        esac
    done
    
    print_info "GGTPassword Android编译脚本"
    print_info "=============================="
    
    if [ "$CLEAN_ONLY" = true ]; then
        clean_build
        exit 0
    fi
    
    # 执行构建流程
    check_environment
    setup_build_config
    configure_cmake
    build_project
    
    if [ "$GENERATE_APK" = true ]; then
        generate_apk
    fi
    
    print_success "构建完成！"
}

# 运行主函数
main "$@"