@echo off
setlocal enabledelayedexpansion

REM Android编译脚本 - Windows版本
REM 用于编译GGTPassword Android版本

REM 设置代码页为UTF-8
chcp 65001 >nul

REM 颜色定义（Windows CMD不支持ANSI颜色，使用echo代替）
set "INFO_PREFIX=[INFO]"
set "SUCCESS_PREFIX=[SUCCESS]"
set "WARNING_PREFIX=[WARNING]"
set "ERROR_PREFIX=[ERROR]"

REM 默认参数
set "BUILD_TYPE=Release"
set "ANDROID_ABI=arm64-v8a"
set "ANDROID_PLATFORM=android-24"
set "BUILD_DIR=build-android-%ANDROID_ABI%"
set "CLEAN_ONLY=false"
set "GENERATE_APK=true"

REM 解析命令行参数
:parse_args
if "%~1"=="" goto :args_done
if /i "%~1"=="-h" goto :show_help
if /i "%~1"=="--help" goto :show_help
if /i "%~1"=="-c" (
    set "CLEAN_ONLY=true"
    shift
    goto :parse_args
)
if /i "%~1"=="--clean" (
    set "CLEAN_ONLY=true"
    shift
    goto :parse_args
)
if /i "%~1"=="-t" (
    set "BUILD_TYPE=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="--type" (
    set "BUILD_TYPE=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="-a" (
    set "ANDROID_ABI=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="--abi" (
    set "ANDROID_ABI=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="-p" (
    set "ANDROID_PLATFORM=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="--platform" (
    set "ANDROID_PLATFORM=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="-d" (
    set "BUILD_DIR=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="--dir" (
    set "BUILD_DIR=%~2"
    shift
    shift
    goto :parse_args
)
if /i "%~1"=="--no-apk" (
    set "GENERATE_APK=false"
    shift
    goto :parse_args
)
echo %ERROR_PREFIX% 未知选项: %~1
goto :show_help

:args_done

echo %INFO_PREFIX% GGTPassword Android编译脚本
echo %INFO_PREFIX% ==============================

if "%CLEAN_ONLY%"=="true" (
    call :clean_build
    exit /b 0
)

REM 执行构建流程
call :check_environment
if errorlevel 1 exit /b 1

call :setup_build_config
if errorlevel 1 exit /b 1

call :configure_cmake
if errorlevel 1 exit /b 1

call :build_project
if errorlevel 1 exit /b 1

if "%GENERATE_APK%"=="true" (
    call :generate_apk
    if errorlevel 1 exit /b 1
)

echo %SUCCESS_PREFIX% 构建完成！
exit /b 0

REM 检查环境函数
:check_environment
echo %INFO_PREFIX% 检查Android开发环境...

REM 检查Android SDK
if "%ANDROID_SDK_ROOT%"=="" if "%ANDROID_HOME%"=="" (
    echo %ERROR_PREFIX% 未设置ANDROID_SDK_ROOT或ANDROID_HOME环境变量
    echo %INFO_PREFIX% 请设置Android SDK路径，例如：
    echo %INFO_PREFIX% set ANDROID_SDK_ROOT=C:\Android\Sdk
    exit /b 1
)

REM 使用ANDROID_SDK_ROOT，如果未设置则使用ANDROID_HOME
if not "%ANDROID_SDK_ROOT%"=="" (
    set "ANDROID_SDK=%ANDROID_SDK_ROOT%"
) else (
    set "ANDROID_SDK=%ANDROID_HOME%"
)

REM 检查Android NDK
if "%ANDROID_NDK_ROOT%"=="" (
    REM 尝试从SDK中查找NDK
    if exist "%ANDROID_SDK%\ndk" (
        for /d %%i in ("%ANDROID_SDK%\ndk\*") do (
            set "ANDROID_NDK_ROOT=%%i"
        )
        if not "!ANDROID_NDK_ROOT!"=="" (
            echo %INFO_PREFIX% 找到NDK: !ANDROID_NDK_ROOT!
        )
    )
    
    if "%ANDROID_NDK_ROOT%"=="" (
        echo %ERROR_PREFIX% 未找到Android NDK
        echo %INFO_PREFIX% 请安装Android NDK或设置ANDROID_NDK_ROOT环境变量
        exit /b 1
    )
)

REM 检查Qt for Android
if "%QT_ANDROID_ROOT%"=="" (
    echo %ERROR_PREFIX% 未设置QT_ANDROID_ROOT环境变量
    echo %INFO_PREFIX% 请设置Qt for Android的安装路径，例如：
    echo %INFO_PREFIX% set QT_ANDROID_ROOT=C:\Qt\6.x.x\android_arm64_v8a
    exit /b 1
)

if not exist "%QT_ANDROID_ROOT%" (
    echo %ERROR_PREFIX% Qt Android路径不存在: %QT_ANDROID_ROOT%
    exit /b 1
)

REM 检查CMake
cmake --version >nul 2>&1
if errorlevel 1 (
    echo %ERROR_PREFIX% 未找到CMake
    echo %INFO_PREFIX% 请安装CMake并添加到PATH
    exit /b 1
)

REM 检查Ninja（可选）
ninja --version >nul 2>&1
if not errorlevel 1 (
    set "CMAKE_GENERATOR=Ninja"
    echo %INFO_PREFIX% 使用Ninja构建系统
) else (
    set "CMAKE_GENERATOR=NMake Makefiles"
    echo %INFO_PREFIX% 使用NMake构建系统
)

REM 检查Java
if "%JAVA_HOME%"=="" (
    echo %WARNING_PREFIX% 未设置JAVA_HOME环境变量
    echo %INFO_PREFIX% APK生成可能会失败
)

echo %SUCCESS_PREFIX% 环境检查完成
exit /b 0

REM 设置构建配置函数
:setup_build_config
echo %INFO_PREFIX% 设置构建配置...

REM 更新构建目录名称
set "BUILD_DIR=build-android-%ANDROID_ABI%"

REM 创建构建目录
if exist "%BUILD_DIR%" (
    echo %WARNING_PREFIX% 构建目录已存在: %BUILD_DIR%
    set /p "REPLY=是否清理并重新构建? (y/N): "
    if /i "!REPLY!"=="y" (
        rmdir /s /q "%BUILD_DIR%"
        echo %INFO_PREFIX% 已清理构建目录
    )
)

if not exist "%BUILD_DIR%" mkdir "%BUILD_DIR%"

echo %INFO_PREFIX% 构建配置:
echo %INFO_PREFIX%   构建类型: %BUILD_TYPE%
echo %INFO_PREFIX%   Android ABI: %ANDROID_ABI%
echo %INFO_PREFIX%   Android平台: %ANDROID_PLATFORM%
echo %INFO_PREFIX%   构建目录: %BUILD_DIR%
echo %INFO_PREFIX%   Qt路径: %QT_ANDROID_ROOT%
echo %INFO_PREFIX%   Android SDK: %ANDROID_SDK%
echo %INFO_PREFIX%   Android NDK: %ANDROID_NDK_ROOT%

exit /b 0

REM 配置CMake函数
:configure_cmake
echo %INFO_PREFIX% 配置CMake...

pushd "%BUILD_DIR%"

REM 在配置前生成构建目录下的 Android 包源（避免污染源码）
set "PACKAGE_DIR=%CD%\android-package"
set "SRC_SRC=..\platform\android\src"
set "SRC_DST=%PACKAGE_DIR%\src"
set "MANIFEST_SRC=..\platform\android\AndroidManifest.xml"
set "MANIFEST_DST=%PACKAGE_DIR%\AndroidManifest.xml"
set "ASSETS_SRC=..\web"
set "ASSETS_DST=%PACKAGE_DIR%\assets\web"

echo %INFO_PREFIX% 准备打包源目录: %PACKAGE_DIR%
if not exist "%PACKAGE_DIR%" mkdir "%PACKAGE_DIR%"
if not exist "%SRC_DST%" mkdir "%SRC_DST%"
if not exist "%ASSETS_DST%" mkdir "%ASSETS_DST%"

REM 同步 src
if exist "%SRC_SRC%" (
    echo %INFO_PREFIX% 同步Android src: %SRC_SRC% ^> %SRC_DST%
    xcopy /E /I /Y "%SRC_SRC%" "%SRC_DST%" >nul
) else (
    echo %WARNING_PREFIX% 未找到src目录: %SRC_SRC%
)

REM 同步 Manifest
if exist "%MANIFEST_SRC%" (
    echo %INFO_PREFIX% 复制AndroidManifest: %MANIFEST_SRC% ^> %MANIFEST_DST%
    copy /Y "%MANIFEST_SRC%" "%MANIFEST_DST%" >nul
) else (
    echo %ERROR_PREFIX% 缺少AndroidManifest.xml: %MANIFEST_SRC%
    popd & exit /b 1
)

REM 同步 Web 前端到 assets\web
if exist "%ASSETS_SRC%" (
    echo %INFO_PREFIX% 同步Android assets: %ASSETS_SRC% ^> %ASSETS_DST%
    xcopy /E /I /Y "%ASSETS_SRC%" "%ASSETS_DST%" >nul
) else (
    echo %WARNING_PREFIX% 未找到web目录: %ASSETS_SRC%
)

REM CMake配置参数
set CMAKE_ARGS=-DCMAKE_BUILD_TYPE=%BUILD_TYPE%
set CMAKE_ARGS=%CMAKE_ARGS% -DCMAKE_TOOLCHAIN_FILE=%ANDROID_NDK_ROOT%\build\cmake\android.toolchain.cmake
set CMAKE_ARGS=%CMAKE_ARGS% -DANDROID_ABI=%ANDROID_ABI%
set CMAKE_ARGS=%CMAKE_ARGS% -DANDROID_PLATFORM=%ANDROID_PLATFORM%
set CMAKE_ARGS=%CMAKE_ARGS% -DANDROID_STL=c++_shared
set CMAKE_ARGS=%CMAKE_ARGS% -DCMAKE_PREFIX_PATH=%QT_ANDROID_ROOT%
set CMAKE_ARGS=%CMAKE_ARGS% -DQT_ANDROID_ROOT=%QT_ANDROID_ROOT%
set CMAKE_ARGS=%CMAKE_ARGS% -DANDROID_SDK_ROOT=%ANDROID_SDK%
set CMAKE_ARGS=%CMAKE_ARGS% -DANDROID_NDK_ROOT=%ANDROID_NDK_ROOT%
set CMAKE_ARGS=%CMAKE_ARGS% -DGGT_ANDROID_PACKAGE_SOURCE_DIR=%PACKAGE_DIR%
set CMAKE_ARGS=%CMAKE_ARGS% -G "%CMAKE_GENERATOR%"

REM 添加调试选项
if /i "%BUILD_TYPE%"=="Debug" (
    set CMAKE_ARGS=%CMAKE_ARGS% -DDEBUG_MODE=ON
) else (
    set CMAKE_ARGS=%CMAKE_ARGS% -DDEBUG_MODE=OFF
)

echo %INFO_PREFIX% 执行CMake配置...
cmake %CMAKE_ARGS% ..

if errorlevel 1 (
    echo %ERROR_PREFIX% CMake配置失败
    popd
    exit /b 1
)

echo %SUCCESS_PREFIX% CMake配置完成
popd
exit /b 0

REM 编译项目函数
:build_project
echo %INFO_PREFIX% 开始编译...

pushd "%BUILD_DIR%"

REM 获取CPU核心数用于并行编译
for /f "tokens=2 delims==" %%i in ('wmic cpu get NumberOfLogicalProcessors /value ^| find "="') do set JOBS=%%i
if "%JOBS%"=="" set JOBS=4

echo %INFO_PREFIX% 使用 %JOBS% 个并行任务进行编译

if "%CMAKE_GENERATOR%"=="Ninja" (
    ninja -j %JOBS%
) else (
    nmake
)

if errorlevel 1 (
    echo %ERROR_PREFIX% 编译失败
    popd
    exit /b 1
)

echo %SUCCESS_PREFIX% 编译完成
popd
exit /b 0

REM 生成APK函数
:generate_apk
echo %INFO_PREFIX% 生成APK...

pushd "%BUILD_DIR%"

REM 查找生成的库文件
for /r . %%f in (libGGTPassword*.so) do (
    set "LIB_FILE=%%f"
    goto :lib_found
)

echo %ERROR_PREFIX% 未找到编译生成的库文件
popd
exit /b 1

:lib_found
echo %INFO_PREFIX% 找到库文件: %LIB_FILE%

REM 使用Qt的androiddeployqt工具生成APK
set "ANDROIDDEPLOYQT=%QT_ANDROID_ROOT%\bin\androiddeployqt.exe"

if not exist "%ANDROIDDEPLOYQT%" (
    echo %ERROR_PREFIX% 未找到androiddeployqt工具: %ANDROIDDEPLOYQT%
    popd
    exit /b 1
)

REM 创建 platform\android 目录（如果不存在）
if not exist "..\platform\android" (
    echo %INFO_PREFIX% 创建Android打包目录: ..\platform\android
    mkdir "..\platform\android"
)

REM 如清单缺失则生成基本的AndroidManifest.xml
if not exist "..\platform\android\AndroidManifest.xml" (
    (
        echo ^<?xml version="1.0"?^>
        echo ^<manifest package="com.ggt.ggtpassword" 
        echo           xmlns:android="http://schemas.android.com/apk/res/android" 
        echo           android:versionName="1.0" 
        echo           android:versionCode="1" 
        echo           android:installLocation="auto"^>
        echo.
        echo     ^<uses-permission android:name="android.permission.INTERNET" /^>
        echo     ^<uses-permission android:name="android.permission.ACCESS_NETWORK_STATE" /^>
        echo.
        echo     ^<application android:hardwareAccelerated="true" 
        echo                  android:name="org.qtproject.qt5.android.bindings.QtApplication" 
        echo                  android:label="GGTPassword"
        echo                  android:theme="@android:style/Theme.NoTitleBar.Fullscreen"^>
        echo.
        echo         ^<activity android:configChanges="orientation|uiMode|screenLayout|screenSize|smallestScreenSize|layoutDirection|locale|fontScale|keyboard|keyboardHidden|navigation|mcc|mnc|density"
        echo                   android:name="org.qtproject.qt5.android.bindings.QtActivity"
        echo                   android:label="GGTPassword"
        echo                   android:screenOrientation="unspecified"
        echo                   android:launchMode="singleTop"
        echo                   android:exported="true"^>
        echo             ^<intent-filter^>
        echo                 ^<action android:name="android.intent.action.MAIN"/^>
        echo                 ^<category android:name="android.intent.category.LAUNCHER"/^>
        echo             ^</intent-filter^>
        echo         ^</activity^>
        echo     ^</application^>
        echo.
        echo     ^<uses-sdk android:minSdkVersion="24" android:targetSdkVersion="33"/^>
        echo ^</manifest^>
    ) > "..\platform\android\AndroidManifest.xml"
)

REM 运行androiddeployqt
"%ANDROIDDEPLOYQT%" --input android-GGTPassword-deployment-settings.json --output ..\android --android-platform %ANDROID_PLATFORM% --gradle

if errorlevel 1 (
    echo %ERROR_PREFIX% APK生成失败
    popd
    exit /b 1
)

echo %SUCCESS_PREFIX% APK生成完成

REM 查找生成的APK文件
for /r ..\android %%f in (*.apk) do (
    echo %SUCCESS_PREFIX% APK文件: %%f
    goto :apk_found
)

:apk_found
popd
exit /b 0

REM 清理函数
:clean_build
echo %INFO_PREFIX% 清理构建文件...

if exist "%BUILD_DIR%" (
    rmdir /s /q "%BUILD_DIR%"
    echo %SUCCESS_PREFIX% 已清理构建目录: %BUILD_DIR%
)

if exist "android" (
    rmdir /s /q "android"
    echo %SUCCESS_PREFIX% 已清理Android输出目录
)

exit /b 0

REM 显示帮助信息
:show_help
echo Android编译脚本 - GGTPassword
echo.
echo 用法: %~nx0 [选项]
echo.
echo 选项:
echo   -h, --help              显示此帮助信息
echo   -c, --clean             清理构建文件
echo   -t, --type TYPE         构建类型 (Debug^|Release，默认: Release)
echo   -a, --abi ABI           Android ABI (arm64-v8a^|armeabi-v7a^|x86^|x86_64，默认: arm64-v8a)
echo   -p, --platform PLATFORM Android平台版本 (默认: android-24)
echo   -d, --dir DIR           构建目录 (默认: build-android-%%ABI%%)
echo   --no-apk                只编译，不生成APK
echo.
echo 环境变量:
echo   ANDROID_SDK_ROOT        Android SDK路径
echo   ANDROID_NDK_ROOT        Android NDK路径
echo   QT_ANDROID_ROOT         Qt for Android路径
echo   JAVA_HOME               Java JDK路径
echo.
echo 示例:
echo   %~nx0                      # 使用默认配置编译
echo   %~nx0 -t Debug -a armeabi-v7a  # 编译Debug版本，ARM 32位
echo   %~nx0 --clean              # 清理构建文件
exit /b 0