#!/bin/bash

# GGTPassword 跨平台构建脚本
# 支持 macOS 和 Windows 平台构建

set -e  # 遇到错误立即退出

# 颜色输出
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 打印带颜色的消息
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 显示帮助信息
show_help() {
    echo "GGTPassword 跨平台构建脚本"
    echo ""
    echo "用法: $0 [选项]"
    echo ""
    echo "选项:"
    echo "  -p, --platform PLATFORM  构建平台 (macos|windows|all) [默认: macos]"
    echo "  -c, --config CONFIG       构建配置 (Debug|Release) [默认: Debug]"
    echo "  -j, --jobs JOBS          并行编译任务数 [默认: 自动检测]"
    echo "  --clean                  清理构建目录"
    echo "  --debug-mode             启用调试模式（Safari开发者工具）"
    echo "  -h, --help               显示此帮助信息"
    echo ""
    echo "平台说明:"
    echo "  macos     - 构建 macOS 版本（当前支持）"
    echo "  windows   - 构建 Windows 版本（预留功能）"
    echo "  all       - 构建所有支持的平台"
    echo ""
    echo "示例:"
    echo "  $0                           # 构建 macOS Debug 版本"
    echo "  $0 -p macos -c Release       # 构建 macOS Release 版本"
    echo "  $0 --clean                   # 清理构建目录"
    echo "  $0 -p macos --debug-mode     # 构建带调试功能的版本"
}

# 默认参数
PLATFORM="macos"
CONFIG="Debug"
JOBS=""
CLEAN_BUILD=false
DEBUG_MODE=false
PROJECT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BUILD_DIR="$PROJECT_DIR/build"

# 解析命令行参数
while [[ $# -gt 0 ]]; do
    case $1 in
        -p|--platform)
            PLATFORM="$2"
            shift 2
            ;;
        -c|--config)
            CONFIG="$2"
            shift 2
            ;;
        -j|--jobs)
            JOBS="$2"
            shift 2
            ;;
        --clean)
            CLEAN_BUILD=true
            shift
            ;;
        --debug-mode)
            DEBUG_MODE=true
            shift
            ;;
        -h|--help)
            show_help
            exit 0
            ;;
        *)
            print_error "未知参数: $1"
            show_help
            exit 1
            ;;
    esac
done

# 验证参数
if [[ "$PLATFORM" != "macos" && "$PLATFORM" != "windows" && "$PLATFORM" != "all" ]]; then
    print_error "无效的构建平台: $PLATFORM"
    show_help
    exit 1
fi

if [[ "$CONFIG" != "Debug" && "$CONFIG" != "Release" ]]; then
    print_error "无效的构建配置: $CONFIG"
    show_help
    exit 1
fi

# 自动检测并行任务数
if [[ -z "$JOBS" ]]; then
    if [[ "$OSTYPE" == "darwin"* ]]; then
        JOBS=$(sysctl -n hw.ncpu)
    else
        JOBS=$(nproc 2>/dev/null || echo "4")
    fi
fi

print_info "GGTPassword 构建脚本启动"
print_info "项目目录: $PROJECT_DIR"
print_info "构建平台: $PLATFORM"
print_info "构建配置: $CONFIG"
print_info "并行任务: $JOBS"
print_info "调试模式: $([ "$DEBUG_MODE" = true ] && echo "启用" || echo "禁用")"

# 清理构建目录
if [[ "$CLEAN_BUILD" = true ]]; then
    print_info "清理构建目录..."
    rm -rf "$BUILD_DIR"
    print_success "构建目录已清理"
    exit 0
fi

# 检查基础工具
check_basic_tools() {
    print_info "检查基础构建工具..."
    
    if ! command -v cmake &> /dev/null; then
        print_error "CMake 未安装或不在 PATH 中"
        print_error "请安装 CMake: https://cmake.org/download/"
        exit 1
    fi
    
    local cmake_version=$(cmake --version | head -n1 | cut -d' ' -f3)
    print_info "CMake 版本: $cmake_version"
    
    if ! command -v git &> /dev/null; then
        print_warning "Git 未安装，可能影响版本信息获取"
    fi
    
    print_success "基础工具检查完成"
}

# macOS 构建函数
build_macos() {
    print_info "开始 macOS 构建..."
    
    # 检查 macOS 特定工具
    if [[ "$OSTYPE" != "darwin"* ]]; then
        print_error "当前系统不是 macOS，无法构建 macOS 版本"
        return 1
    fi
    
    if ! command -v xcodebuild &> /dev/null; then
        print_error "Xcode 未安装或不在 PATH 中"
        print_error "请安装 Xcode 或 Xcode Command Line Tools"
        return 1
    fi
    
    # 检查 Qt for macOS
    print_info "检查 Qt for macOS 安装..."
    
    local qt_cmake_dir=""
    local qt_version=""
    
    # 优先使用官方 Qt 安装
    if [[ -d "$HOME/Qt/6.10.0/macos" ]]; then
        qt_cmake_dir="$HOME/Qt/6.10.0/macos/lib/cmake/Qt6"
        qt_version="6.10.0"
        print_info "使用官方 Qt 6.10.0 macOS 安装"
    elif [[ -d "$HOME/Qt/6.9.0/macos" ]]; then
        qt_cmake_dir="$HOME/Qt/6.9.0/macos/lib/cmake/Qt6"
        qt_version="6.9.0"
        print_info "使用官方 Qt 6.9.0 macOS 安装"
    elif [[ -d "/opt/homebrew/lib/cmake/Qt6" ]]; then
        qt_cmake_dir="/opt/homebrew/lib/cmake/Qt6"
        qt_version=$(ls "/opt/homebrew/Cellar/qt" | head -1 2>/dev/null || echo "unknown")
        print_info "使用 Homebrew Qt 安装 (版本: $qt_version)"
    elif [[ -d "/usr/local/lib/cmake/Qt6" ]]; then
        qt_cmake_dir="/usr/local/lib/cmake/Qt6"
        qt_version="system"
        print_info "使用系统 Qt 安装"
    else
        print_error "未找到 Qt for macOS 安装"
        print_error "请安装 Qt for macOS:"
        print_error "  1. 从 https://www.qt.io/download 下载官方安装包"
        print_error "  2. 或使用 Homebrew: brew install qt"
        return 1
    fi
    
    if [[ ! -d "$qt_cmake_dir" ]]; then
        print_error "Qt CMake 目录不存在: $qt_cmake_dir"
        return 1
    fi
    
    print_success "Qt 环境检查完成"
    print_info "Qt CMake 目录: $qt_cmake_dir"
    
    # 创建构建目录
    local macos_build_dir="$BUILD_DIR/macos"
    mkdir -p "$macos_build_dir"
    cd "$macos_build_dir"
    
    print_info "配置 CMake..."
    
    # CMake 配置参数
    local cmake_args=(
        -DCMAKE_BUILD_TYPE="$CONFIG"
        -DCMAKE_PREFIX_PATH="$qt_cmake_dir"
        -DQt6_DIR="$qt_cmake_dir"
        -DCMAKE_OSX_DEPLOYMENT_TARGET="10.15"
    )
    
    # 添加调试模式
    if [[ "$DEBUG_MODE" = true ]]; then
        cmake_args+=(-DDEBUG_MODE=ON)
        print_info "启用调试模式 - Safari 开发者工具将可用"
    else
        cmake_args+=(-DDEBUG_MODE=OFF)
    fi
    
    print_info "CMake 参数: ${cmake_args[*]}"
    
    # 运行 CMake 配置
    cmake "${cmake_args[@]}" "$PROJECT_DIR"
    
    if [[ $? -ne 0 ]]; then
        print_error "CMake 配置失败"
        return 1
    fi
    
    print_info "开始编译..."
    
    # 编译项目
    cmake --build . --config "$CONFIG" --parallel "$JOBS"
    
    if [[ $? -ne 0 ]]; then
        print_error "编译失败"
        return 1
    fi
    
    print_success "macOS 构建完成"
    
    # 显示输出文件位置
    local app_path="$macos_build_dir/GGTPassword.app"
    if [[ -d "$app_path" ]]; then
        print_success "应用包位置: $app_path"
        print_info "运行应用: open \"$app_path\""
    fi
    
    return 0
}

# Windows 构建函数（预留）
build_windows() {
    print_warning "Windows 构建功能尚未实现"
    print_info "预留功能说明:"
    print_info "  - 将支持 MSVC 和 MinGW 编译器"
    print_info "  - 将支持 Qt for Windows"
    print_info "  - 将支持 Visual Studio 项目生成"
    print_info "  - 将支持 Windows 特定的 WebView2 集成"
    
    # TODO: 实现 Windows 构建逻辑
    # 1. 检查 Windows 构建环境（MSVC/MinGW）
    # 2. 检查 Qt for Windows 安装
    # 3. 配置 CMake for Windows
    # 4. 编译 Windows 版本
    # 5. 生成安装包
    
    return 1
}

# 主构建逻辑
main() {
    check_basic_tools
    
    local build_success=true
    
    case "$PLATFORM" in
        "macos")
            build_macos || build_success=false
            ;;
        "windows")
            build_windows || build_success=false
            ;;
        "all")
            print_info "构建所有支持的平台..."
            build_macos || build_success=false
            
            if [[ "$build_success" = true ]]; then
                print_info "尝试构建 Windows 版本..."
                build_windows || print_warning "Windows 构建跳过（功能未实现）"
            fi
            ;;
    esac
    
    if [[ "$build_success" = true ]]; then
        print_success "构建任务完成！"
        
        echo ""
        print_info "下一步操作:"
        if [[ "$PLATFORM" == "macos" || "$PLATFORM" == "all" ]]; then
            echo "  • 运行应用: open \"$BUILD_DIR/macos/GGTPassword.app\""
            echo "  • 调试应用: lldb \"$BUILD_DIR/macos/GGTPassword.app/Contents/MacOS/GGTPassword\""
        fi
        echo "  • 清理构建: $0 --clean"
        echo "  • 重新构建: $0 -p $PLATFORM -c $CONFIG"
    else
        print_error "构建失败！"
        exit 1
    fi
}

# 执行主函数
cd "$PROJECT_DIR"
main