@echo off
setlocal EnableExtensions EnableDelayedExpansion

rem ========================================
rem Unified Build Script for GGTPassword
rem Usage:
rem   build.bat [app|test] [Debug|Release] [--clean] [--run] [--deploy] [--no-prompt]
rem   build.bat --help
rem Defaults:
rem   mode=app, config=Debug, no clean, no run, no deploy
rem ========================================

set "MODE=app"
set "CONFIG=Debug"
set "CLEAN=0"
set "RUN=0"
set "DEPLOY=0"
set "NOPROMPT=0"
set "HELP=0"

rem ---- parse first positional arg (mode) ----
if not "%~1"=="" (
  set "arg=%~1"
  if /i "%arg%"=="app"  ( set "MODE=app"  & shift ) else (
  if /i "%arg%"=="test" ( set "MODE=test" & shift ) )
)

rem ---- parse second positional arg (config) ----
if not "%~1"=="" (
  set "arg=%~1"
  if /i "%arg%"=="debug"   ( set "CONFIG=Debug"   & shift ) else (
  if /i "%arg%"=="release" ( set "CONFIG=Release" & shift ) )
)

rem ---- parse flags ----
:parse_flags
if "%~1"=="" goto flags_done
set "arg=%~1"
if /i "%arg%"=="--help"      set "HELP=1"& shift & goto parse_flags
if /i "%arg%"=="-h"          set "HELP=1"& shift & goto parse_flags
if /i "%arg%"=="/?"          set "HELP=1"& shift & goto parse_flags
if /i "%arg%"=="--clean"     set "CLEAN=1"& shift & goto parse_flags
if /i "%arg%"=="--run"       set "RUN=1"  & shift & goto parse_flags
if /i "%arg%"=="--deploy"    set "DEPLOY=1"& shift & goto parse_flags
if /i "%arg%"=="--no-prompt" set "NOPROMPT=1"& shift & goto parse_flags
shift
goto parse_flags
:flags_done

if "%HELP%"=="1" goto :usage

echo ========================================
echo Unified Build - mode:%MODE% config:%CONFIG% clean:%CLEAN% run:%RUN% deploy:%DEPLOY%
echo ========================================

rem ---- environment checks ----
echo Checking Qt toolchain (qmake)...
where qmake >nul 2>&1
if %errorlevel% neq 0 (
  echo ERROR: Qt not found in PATH. Example: set PATH=C:\Qt\6.10.0\msvc2022_64\bin;%%PATH%%
  exit /b 1
)

echo Checking CMake...
where cmake >nul 2>&1
if %errorlevel% neq 0 (
  echo ERROR: CMake not found. Install from https://cmake.org/download/
  exit /b 1
)

rem ---- decide build dir, cmake options, target, exe ----
set "BUILD_DIR=build_windows"
set "CMAKE_EXTRA="
set "BUILD_TARGET="
set "RUN_EXE="

if /i "%MODE%"=="app" (
  set "BUILD_DIR=build_windows"
  set "CMAKE_EXTRA="
  set "BUILD_TARGET=--config %CONFIG%"
  set "RUN_EXE=%BUILD_DIR%\%CONFIG%\GGTPassword.exe"
) else (
  set "BUILD_DIR=build_test"
  set "CMAKE_EXTRA=-DBUILD_TEST_ONLY=ON"
  set "BUILD_TARGET=--config %CONFIG% --target test_webview_simple"
  set "RUN_EXE=%BUILD_DIR%\%CONFIG%\test_webview_simple.exe"
)

rem ---- clean if requested ----
if "%CLEAN%"=="1" (
  if exist "%BUILD_DIR%" (
    echo Cleaning build directory: %BUILD_DIR%
    rmdir /s /q "%BUILD_DIR%"
  )
)

rem ---- configure & build ----
if not exist "%BUILD_DIR%" mkdir "%BUILD_DIR%"
pushd "%BUILD_DIR%"

echo Configuring CMake project...
cmake .. -G "Visual Studio 17 2022" -A x64 -DCMAKE_BUILD_TYPE=%CONFIG% %CMAKE_EXTRA%
if %errorlevel% neq 0 (
  popd
  echo ERROR: CMake configuration failed
  exit /b 1
)

echo Building project...
cmake --build . %BUILD_TARGET%
if %errorlevel% neq 0 (
  popd
  echo ERROR: Build failed
  exit /b 1
)

popd

echo ========================================
echo Build completed successfully
if exist "%RUN_EXE%" (
  echo Executable: %RUN_EXE%
) else (
  echo NOTE: Executable not found at %RUN_EXE%
)
echo ========================================

rem ---- optional deploy for app ----
if "%DEPLOY%"=="1" (
  if /i "%MODE%"=="app" (
    if exist "%~dp0deploy_qt_dlls.bat" (
      echo Running deploy script...
      call "%~dp0deploy_qt_dlls.bat"
    ) else (
      echo WARNING: deploy_qt_dlls.bat not found. Skipping deploy.
    )
  ) else (
    echo INFO: --deploy ignored for test mode.
  )
)

rem ---- run or prompt ----
if "%RUN%"=="1" goto :run_now
if "%NOPROMPT%"=="1" goto :end
if exist "%RUN_EXE%" (
  set /p __RUNNOW="Run now? (y/n): "
  if /i "%__RUNNOW%"=="y" goto :run_now
)

goto :end

:run_now
if not exist "%RUN_EXE%" (
  echo ERROR: Executable not found: %RUN_EXE%
  goto :end
)
echo Starting: %RUN_EXE%
start "" "%RUN_EXE%"

goto :end

:usage
echo.
echo Usage:
echo   build.bat [app^|test] [Debug^|Release] [--clean] [--run] [--deploy] [--no-prompt]
echo Examples:
echo   build.bat                              ^(default: app Debug^)
echo   build.bat app Release --clean --run
echo   build.bat test Debug --clean --run
exit /b 0

:end
exit /b 0