#ifndef WEBVIEWBRIDGE_H
#define WEBVIEWBRIDGE_H

#include <QObject>
#include <QString>
#include <QJsonObject>
#include <QJsonDocument>
#include <functional>

/**
 * @brief WebView通信桥接基类
 * 定义了跨平台WebView与C++通信的统一接口
 */
class WebViewBridge : public QObject
{
    Q_OBJECT

public:
    explicit WebViewBridge(QObject *parent = nullptr);
    virtual ~WebViewBridge() = default;

    // 消息处理回调函数类型
    using MessageHandler = std::function<void(const QString& method, const QJsonObject& params)>;

    /**
     * @brief 设置消息处理器
     * @param handler 处理从WebView发来消息的回调函数
     */
    void setMessageHandler(MessageHandler handler);

    /**
     * @brief 向WebView发送消息
     * @param method 方法名
     * @param params 参数对象
     */
    virtual void sendMessage(const QString& method, const QJsonObject& params = QJsonObject()) = 0;

    /**
     * @brief 执行JavaScript代码
     * @param script JavaScript代码
     */
    virtual void evaluateJavaScript(const QString& script) = 0;

    /**
     * @brief 初始化WebView
     */
    virtual void initializeWebView() = 0;

    /**
     * @brief 加载HTML内容
     * @param html HTML内容
     */
    virtual void loadHtml(const QString& html) = 0;

    /**
     * @brief 加载URL
     * @param url 要加载的URL
     */
    virtual void loadUrl(const QString& url) = 0;

protected:
    /**
     * @brief 处理从WebView接收到的消息
     * @param message JSON格式的消息字符串
     */
    void handleMessage(const QString& message);

    /**
     * @brief 创建标准的响应消息
     * @param method 方法名
     * @param data 响应数据
     * @param success 是否成功
     * @param error 错误信息（如果有）
     */
    QJsonObject createResponse(const QString& method, const QJsonObject& data = QJsonObject(), 
                              bool success = true, const QString& error = QString());

signals:
    /**
     * @brief WebView准备就绪信号
     */
    void webViewReady();

    /**
     * @brief 接收到消息信号
     * @param method 方法名
     * @param params 参数
     */
    void messageReceived(const QString& method, const QJsonObject& params);

private:
    MessageHandler m_messageHandler;
};

#endif // WEBVIEWBRIDGE_H